% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_matrix.R
\name{kernel_matrix}
\alias{kernel_matrix}
\title{Compute Kernel Matrix Between Input Locations}
\usage{
kernel_matrix(
  X,
  Xprime = NULL,
  theta = 0.1,
  kernel = c("gaussian", "matern52", "matern32"),
  anisotropic = TRUE
)
}
\arguments{
\item{X}{A numeric matrix (or vector) of input locations with shape \eqn{n
\times d}.}

\item{Xprime}{An optional numeric matrix of input locations with shape \eqn{m
  \times d}. If \code{NULL} (default), it is set to \code{X}, resulting in a
symmetric matrix.}

\item{theta}{A positive numeric value or vector specifying the kernel
lengthscale(s). If a scalar, the same lengthscale is applied to all input
dimensions. If a vector, it must be of length \code{d}, corresponding to
anisotropic scaling.}

\item{kernel}{A character string specifying the kernel function. Must be one
of \code{"gaussian"}, \code{"matern32"}, or \code{"matern52"}.}

\item{anisotropic}{Logical. If \code{TRUE} (default), \code{theta} is
interpreted as a vector of per-dimension lengthscales. If \code{FALSE},
\code{theta} is treated as a scalar.}
}
\value{
A numeric matrix of size \eqn{n \times m}, where each element
\eqn{K_{ij}} gives the kernel similarity between input \eqn{X_i} and
\eqn{X'_j}.
}
\description{
Computes the kernel matrix between two sets of input locations
using a specified kernel function. Supports both isotropic and anisotropic
lengthscales. Available kernels include the Gaussian, Matérn 5/2, and
Matérn 3/2.
}
\details{
Let \eqn{\mathbf{x}} and \eqn{\mathbf{x}'} denote two input points.
The scaled distance is defined as
\deqn{
     r = \left\| \frac{\mathbf{x} - \mathbf{x}'}{\boldsymbol{\theta}} \right\|_2.
  }

The available kernels are defined as:
\itemize{
\item \strong{Gaussian:}
\deqn{
       k(\mathbf{x}, \mathbf{x}') = \exp(-r^2)
     }
\item \strong{Matérn 5/2:}
\deqn{
       k(\mathbf{x}, \mathbf{x}') = \left(1 + \sqrt{5} r + \frac{5}{3} r^2 \right) \exp(-\sqrt{5} r)
     }
\item \strong{Matérn 3/2:}
\deqn{
       k(\mathbf{x}, \mathbf{x}') = \left(1 + \sqrt{3} r \right) \exp(-\sqrt{3} r)
     }
}

The function performs consistency checks on input dimensions and
automatically broadcasts \code{theta} when it is a scalar.
}
\examples{
# Basic usage with default Xprime = X
X <- matrix(runif(20), ncol = 2)
K1 <- kernel_matrix(X, theta = 0.2, kernel = "gaussian")

# Anisotropic lengthscales with Matérn 5/2
K2 <- kernel_matrix(X, theta = c(0.1, 0.3), kernel = "matern52")

# Isotropic Matérn 3/2
K3 <- kernel_matrix(X, theta = 1, kernel = "matern32", anisotropic = FALSE)

# Use Xprime different from X
Xprime <- matrix(runif(10), ncol = 2)
K4 <- kernel_matrix(X, Xprime, theta = 0.2, kernel = "gaussian")

}
\references{
Rasmussen, C. E., & Williams, C. K. I. (2006). \emph{Gaussian
Processes for Machine Learning}. MIT Press.
}
