#' @title Run Standard CIF Analysis
#' @description Given a prepped data list from \code{\link{prep_data_cif}}, run the standard CIF analysis.
#' @importFrom survival survfit Surv
#' @importFrom stats var cov pnorm
#' @param prepped A list returned by \code{prep_data_cif()}, containing \code{Treatment} and \code{Control}.
#' @param tau Numeric, time horizon (e.g. 15 or 29).
#'
#' @details
#' - RMLT1 uses parameters \code{(a,b,c) = (0,1,0)} for recovery/discharge analysis.
#' - RMLT2 uses \code{(a,b,c) = (0,0,1)} for death analysis.
#'
#' @return A list with formatted results for RMLT1 and RMLT2.
#' @export

do_cif_analysis <- function(prepped, tau=15){
  Treatment <- prepped$Treatment
  Control <- prepped$Control

  rmlt1_tab <- table1_cif(Treatment, Control, tau, 0, 1, 0)
  rmlt2_tab <- table1_cif(Treatment, Control, tau, 0, 0, 1)

  format_cif_results <- function(mat, analysis_type) {
    # Calculate standard errors from confidence intervals
    se_group1 <- (mat[1,3] - mat[1,2]) / (2 * 1.96)
    se_group2 <- (mat[2,3] - mat[2,2]) / (2 * 1.96)

    group_results <- data.frame(
      Group = c("Group1 (trt=1)", "Group2 (trt=0)"),
      Estimate = c(mat[1,1], mat[2,1]),
      se = c(se_group1, se_group2),
      Lower95 = c(mat[1,2], mat[2,2]),
      Upper95 = c(mat[1,3], mat[2,3]),
      stringsAsFactors = FALSE
    )

    contrast_se <- (mat[3,3] - mat[3,2]) / (2 * 1.96)
    contrast_results <- data.frame(
      Contrast = "Difference",
      Estimate = mat[3,1],
      se = contrast_se,
      Lower95 = mat[3,2],
      Upper95 = mat[3,3],
      p_value = mat[3,4],
      stringsAsFactors = FALSE
    )
    list(
      analysis_type = analysis_type,
      groups = group_results,
      contrast = contrast_results
    )
  }

  formatted_rmlt1 <- format_cif_results(rmlt1_tab, "RMLT1")
  formatted_rmlt2 <- format_cif_results(rmlt2_tab, "RMLT2")

  return(list(
    RMLT1 = formatted_rmlt1,
    RMLT2 = formatted_rmlt2
  ))
}
