/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.fate.zookeeper;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.accumulo.core.fate.zookeeper.ServiceLock;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooCache {
    private static final Logger log = LoggerFactory.getLogger(ZooCache.class);
    private final ZCacheWatcher watcher = new ZCacheWatcher();
    private final Watcher externalWatcher;
    private final ReadWriteLock cacheLock = new ReentrantReadWriteLock(false);
    private final Lock cacheWriteLock = this.cacheLock.writeLock();
    private final Lock cacheReadLock = this.cacheLock.readLock();
    private final HashMap<String, byte[]> cache;
    private final HashMap<String, ZcStat> statCache;
    private final HashMap<String, List<String>> childrenCache;
    private final ZooReader zReader;
    private static final SecureRandom random = new SecureRandom();
    private volatile boolean closed = false;
    private volatile ImmutableCacheCopies immutableCache = new ImmutableCacheCopies(0L);
    private long updateCount = 0L;

    private ZooKeeper getZooKeeper() {
        return this.zReader.getZooKeeper();
    }

    public ZooCache(ZooReader reader, Watcher watcher) {
        this.zReader = reader;
        this.cache = new HashMap();
        this.statCache = new HashMap();
        this.childrenCache = new HashMap();
        this.externalWatcher = watcher;
    }

    public List<String> getChildren(final String zPath) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        ZooRunnable<List<String>> zr = new ZooRunnable<List<String>>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public List<String> run() throws KeeperException, InterruptedException {
                ImmutableCacheCopies lic = ZooCache.this.immutableCache;
                if (lic.childrenCache.containsKey(zPath)) {
                    return lic.childrenCache.get(zPath);
                }
                ZooCache.this.cacheWriteLock.lock();
                try {
                    if (ZooCache.this.childrenCache.containsKey(zPath)) {
                        List<String> list = ZooCache.this.childrenCache.get(zPath);
                        return list;
                    }
                    ZooKeeper zooKeeper = ZooCache.this.getZooKeeper();
                    List<String> children = zooKeeper.getChildren(zPath, (Watcher)ZooCache.this.watcher);
                    if (children != null) {
                        children = List.copyOf(children);
                    }
                    ZooCache.this.childrenCache.put(zPath, children);
                    ZooCache.this.immutableCache = new ImmutableCacheCopies(++ZooCache.this.updateCount, ZooCache.this.immutableCache, ZooCache.this.childrenCache);
                    List<String> list = children;
                    return list;
                }
                catch (KeeperException ke) {
                    if (ke.code() != KeeperException.Code.NONODE) {
                        throw ke;
                    }
                }
                finally {
                    ZooCache.this.cacheWriteLock.unlock();
                }
                return null;
            }
        };
        return (List)zr.retry();
    }

    public byte[] get(String zPath) {
        return this.get(zPath, null);
    }

    public byte[] get(final String zPath, final ZcStat status) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        ZooRunnable<byte[]> zr = new ZooRunnable<byte[]>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public byte[] run() throws KeeperException, InterruptedException {
                ZcStat zstat = null;
                ImmutableCacheCopies lic = ZooCache.this.immutableCache;
                byte[] val = lic.cache.get(zPath);
                if (val != null || lic.cache.containsKey(zPath)) {
                    if (status != null) {
                        zstat = lic.statCache.get(zPath);
                        ZooCache.this.copyStats(status, zstat);
                    }
                    return val;
                }
                ZooCache.this.cacheWriteLock.lock();
                try {
                    ZooKeeper zooKeeper = ZooCache.this.getZooKeeper();
                    Stat stat = zooKeeper.exists(zPath, (Watcher)ZooCache.this.watcher);
                    byte[] data = null;
                    if (stat == null) {
                        if (log.isTraceEnabled()) {
                            log.trace("zookeeper did not contain {}", (Object)zPath);
                        }
                    } else {
                        try {
                            data = zooKeeper.getData(zPath, (Watcher)ZooCache.this.watcher, stat);
                            zstat = new ZcStat(stat);
                        }
                        catch (KeeperException.BadVersionException | KeeperException.NoNodeException e1) {
                            throw new ConcurrentModificationException();
                        }
                        if (log.isTraceEnabled()) {
                            log.trace("zookeeper contained {} {}", (Object)zPath, (Object)(data == null ? null : new String(data, StandardCharsets.UTF_8)));
                        }
                    }
                    ZooCache.this.put(zPath, data, zstat);
                    ZooCache.this.copyStats(status, zstat);
                    byte[] byArray = data;
                    return byArray;
                }
                finally {
                    ZooCache.this.cacheWriteLock.unlock();
                }
            }
        };
        return (byte[])zr.retry();
    }

    protected void copyStats(ZcStat userStat, ZcStat cachedStat) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        if (userStat != null && cachedStat != null) {
            userStat.set(cachedStat);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void put(String zPath, byte[] data, ZcStat stat) {
        this.cacheWriteLock.lock();
        try {
            this.cache.put(zPath, data);
            this.statCache.put(zPath, stat);
            this.immutableCache = new ImmutableCacheCopies(++this.updateCount, this.cache, this.statCache, this.immutableCache);
        }
        finally {
            this.cacheWriteLock.unlock();
        }
    }

    private void remove(String zPath) {
        this.cacheWriteLock.lock();
        try {
            this.cache.remove(zPath);
            this.childrenCache.remove(zPath);
            this.statCache.remove(zPath);
            this.immutableCache = new ImmutableCacheCopies(++this.updateCount, this.cache, this.statCache, this.childrenCache);
        }
        finally {
            this.cacheWriteLock.unlock();
        }
    }

    public void clear() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        this.cacheWriteLock.lock();
        try {
            this.cache.clear();
            this.childrenCache.clear();
            this.statCache.clear();
            this.immutableCache = new ImmutableCacheCopies(++this.updateCount);
        }
        finally {
            this.cacheWriteLock.unlock();
        }
    }

    public void close() {
        this.closed = true;
    }

    public long getUpdateCount() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        return this.immutableCache.updateCount;
    }

    @VisibleForTesting
    boolean dataCached(String zPath) {
        this.cacheReadLock.lock();
        try {
            boolean bl = this.immutableCache.cache.containsKey(zPath) && this.cache.containsKey(zPath);
            return bl;
        }
        finally {
            this.cacheReadLock.unlock();
        }
    }

    @VisibleForTesting
    boolean childrenCached(String zPath) {
        this.cacheReadLock.lock();
        try {
            boolean bl = this.immutableCache.childrenCache.containsKey(zPath) && this.childrenCache.containsKey(zPath);
            return bl;
        }
        finally {
            this.cacheReadLock.unlock();
        }
    }

    public void clear(String zPath) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        this.cacheWriteLock.lock();
        try {
            this.cache.keySet().removeIf(path -> path.startsWith(zPath));
            this.childrenCache.keySet().removeIf(path -> path.startsWith(zPath));
            this.statCache.keySet().removeIf(path -> path.startsWith(zPath));
            this.immutableCache = new ImmutableCacheCopies(++this.updateCount, this.cache, this.statCache, this.childrenCache);
        }
        finally {
            this.cacheWriteLock.unlock();
        }
    }

    public byte[] getLockData(ServiceLock.ServiceLockPath path) {
        List<String> children = ServiceLock.validateAndSort(path, this.getChildren(path.toString()));
        if (children == null || children.isEmpty()) {
            return null;
        }
        String lockNode = children.get(0);
        return this.get(path + "/" + lockNode);
    }

    private class ZCacheWatcher
    implements Watcher {
        private ZCacheWatcher() {
        }

        public void process(WatchedEvent event) {
            if (log.isTraceEnabled()) {
                log.trace("{}", (Object)event);
            }
            block0 : switch (event.getType()) {
                case NodeDataChanged: 
                case NodeChildrenChanged: 
                case NodeCreated: 
                case NodeDeleted: {
                    ZooCache.this.remove(event.getPath());
                    break;
                }
                case None: {
                    switch (event.getState()) {
                        case Closed: {
                            log.trace("ZooKeeper connection closed, ignoring; {}", (Object)event);
                            break block0;
                        }
                        case Disconnected: {
                            log.trace("ZooKeeper connection disconnected, clearing cache; {}", (Object)event);
                            ZooCache.this.clear();
                            break block0;
                        }
                        case SyncConnected: {
                            log.trace("ZooKeeper connection established, ignoring; {}", (Object)event);
                            break block0;
                        }
                        case Expired: {
                            log.trace("ZooKeeper connection expired, clearing cache; {}", (Object)event);
                            ZooCache.this.clear();
                            break block0;
                        }
                    }
                    log.warn("Unhandled {}", (Object)event);
                    break;
                }
                default: {
                    log.warn("Unhandled {}", (Object)event);
                }
            }
            if (ZooCache.this.externalWatcher != null) {
                ZooCache.this.externalWatcher.process(event);
            }
        }
    }

    private static class ImmutableCacheCopies {
        final Map<String, byte[]> cache;
        final Map<String, ZcStat> statCache;
        final Map<String, List<String>> childrenCache;
        final long updateCount;

        ImmutableCacheCopies(long updateCount) {
            this.updateCount = updateCount;
            this.cache = Collections.emptyMap();
            this.statCache = Collections.emptyMap();
            this.childrenCache = Collections.emptyMap();
        }

        ImmutableCacheCopies(long updateCount, Map<String, byte[]> cache, Map<String, ZcStat> statCache, Map<String, List<String>> childrenCache) {
            this.updateCount = updateCount;
            this.cache = Collections.unmodifiableMap(new HashMap<String, byte[]>(cache));
            this.statCache = Collections.unmodifiableMap(new HashMap<String, ZcStat>(statCache));
            this.childrenCache = Collections.unmodifiableMap(new HashMap<String, List<String>>(childrenCache));
        }

        ImmutableCacheCopies(long updateCount, ImmutableCacheCopies prev, Map<String, List<String>> childrenCache) {
            this.updateCount = updateCount;
            this.cache = prev.cache;
            this.statCache = prev.statCache;
            this.childrenCache = Collections.unmodifiableMap(new HashMap<String, List<String>>(childrenCache));
        }

        ImmutableCacheCopies(long updateCount, Map<String, byte[]> cache, Map<String, ZcStat> statCache, ImmutableCacheCopies prev) {
            this.updateCount = updateCount;
            this.cache = Collections.unmodifiableMap(new HashMap<String, byte[]>(cache));
            this.statCache = Collections.unmodifiableMap(new HashMap<String, ZcStat>(statCache));
            this.childrenCache = prev.childrenCache;
        }
    }

    private static abstract class ZooRunnable<T> {
        private ZooRunnable() {
        }

        abstract T run() throws KeeperException, InterruptedException;

        public T retry() {
            int sleepTime = 100;
            while (true) {
                try {
                    return this.run();
                }
                catch (KeeperException e) {
                    KeeperException.Code code = e.code();
                    if (code == KeeperException.Code.NONODE) {
                        log.error("Looked up non-existent node in cache " + e.getPath(), (Throwable)e);
                    } else if (code == KeeperException.Code.CONNECTIONLOSS || code == KeeperException.Code.OPERATIONTIMEOUT || code == KeeperException.Code.SESSIONEXPIRED) {
                        log.warn("Saw (possibly) transient exception communicating with ZooKeeper, will retry", (Throwable)e);
                    } else {
                        log.warn("Zookeeper error, will retry", (Throwable)e);
                    }
                }
                catch (InterruptedException e) {
                    log.info("Zookeeper error, will retry", (Throwable)e);
                }
                catch (ConcurrentModificationException e) {
                    log.debug("Zookeeper was modified, will retry");
                }
                try {
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException e) {
                    log.debug("Wait in retry() was interrupted.", (Throwable)e);
                }
                LockSupport.parkNanos(sleepTime);
                if (sleepTime >= 10000) continue;
                sleepTime = (int)((double)sleepTime + (double)sleepTime * random.nextDouble());
            }
        }
    }

    public static class ZcStat {
        private long ephemeralOwner;
        private long mzxid;

        public ZcStat() {
        }

        private ZcStat(Stat stat) {
            this.ephemeralOwner = stat.getEphemeralOwner();
            this.mzxid = stat.getMzxid();
        }

        public long getEphemeralOwner() {
            return this.ephemeralOwner;
        }

        private void set(ZcStat cachedStat) {
            this.ephemeralOwner = cachedStat.ephemeralOwner;
            this.mzxid = cachedStat.mzxid;
        }

        @VisibleForTesting
        public void setEphemeralOwner(long ephemeralOwner) {
            this.ephemeralOwner = ephemeralOwner;
        }

        public long getMzxid() {
            return this.mzxid;
        }
    }
}

