/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.file.blockfile.cache.tinylfu;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Policy;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.accumulo.core.file.blockfile.cache.impl.ClassSize;
import org.apache.accumulo.core.spi.cache.BlockCache;
import org.apache.accumulo.core.spi.cache.BlockCacheManager;
import org.apache.accumulo.core.spi.cache.CacheEntry;
import org.apache.accumulo.core.spi.cache.CacheType;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TinyLfuBlockCache
implements BlockCache {
    private static final Logger log = LoggerFactory.getLogger(TinyLfuBlockCache.class);
    private static final int STATS_PERIOD_SEC = 60;
    private final Cache<String, Block> cache;
    private final Policy.Eviction<String, Block> policy;
    private final int maxSize;
    private final ScheduledExecutorService statsExecutor = ThreadPools.getServerThreadPools().createScheduledExecutorService(1, "TinyLfuBlockCacheStatsExecutor", true);

    public TinyLfuBlockCache(BlockCacheManager.Configuration conf, CacheType type) {
        this.cache = Caffeine.newBuilder().initialCapacity((int)Math.ceil(1.2 * (double)conf.getMaxSize(type) / (double)conf.getBlockSize())).weigher((blockName, block) -> {
            int keyWeight = ClassSize.align(blockName.length()) + ClassSize.STRING;
            return keyWeight + block.weight();
        }).maximumWeight(conf.getMaxSize(type)).recordStats().build();
        this.policy = (Policy.Eviction)this.cache.policy().eviction().get();
        this.maxSize = (int)Math.min(Integer.MAX_VALUE, this.policy.getMaximum());
        ScheduledFuture<?> future = this.statsExecutor.scheduleAtFixedRate(this::logStats, 60L, 60L, TimeUnit.SECONDS);
        ThreadPools.watchNonCriticalScheduledTask(future);
    }

    @Override
    public long getMaxHeapSize() {
        return this.getMaxSize();
    }

    @Override
    public long getMaxSize() {
        return this.maxSize;
    }

    @Override
    public CacheEntry getBlock(String blockName) {
        return this.wrap(blockName, (Block)this.cache.getIfPresent((Object)blockName));
    }

    @Override
    public CacheEntry cacheBlock(String blockName, byte[] buffer) {
        return this.wrap(blockName, this.cache.asMap().compute(blockName, (key, block) -> new Block(buffer)));
    }

    @Override
    public BlockCache.Stats getStats() {
        final CacheStats stats = this.cache.stats();
        return new BlockCache.Stats(){

            @Override
            public long hitCount() {
                return stats.hitCount();
            }

            @Override
            public long requestCount() {
                return stats.requestCount();
            }
        };
    }

    private void logStats() {
        double maxMB = (double)this.policy.getMaximum() / 1048576.0;
        double sizeMB = (double)this.policy.weightedSize().getAsLong() / 1048576.0;
        double freeMB = maxMB - sizeMB;
        log.debug("Cache Size={}MB, Free={}MB, Max={}MB, Blocks={}", new Object[]{sizeMB, freeMB, maxMB, this.cache.estimatedSize()});
        log.debug(this.cache.stats().toString());
    }

    private CacheEntry wrap(String cacheKey, Block block) {
        if (block != null) {
            return new TlfuCacheEntry(cacheKey, block);
        }
        return null;
    }

    private Block load(BlockCache.Loader loader, Map<String, byte[]> resolvedDeps) {
        byte[] data = loader.load(this.maxSize, resolvedDeps);
        return data == null ? null : new Block(data);
    }

    private Map<String, byte[]> resolveDependencies(Map<String, BlockCache.Loader> deps) {
        if (deps.size() == 1) {
            Map.Entry<String, BlockCache.Loader> entry = deps.entrySet().iterator().next();
            CacheEntry ce = this.getBlock(entry.getKey(), entry.getValue());
            if (ce == null) {
                return null;
            }
            return Collections.singletonMap(entry.getKey(), ce.getBuffer());
        }
        HashMap<String, byte[]> resolvedDeps = new HashMap<String, byte[]>();
        for (Map.Entry<String, BlockCache.Loader> entry : deps.entrySet()) {
            CacheEntry ce = this.getBlock(entry.getKey(), entry.getValue());
            if (ce == null) {
                return null;
            }
            resolvedDeps.put(entry.getKey(), ce.getBuffer());
        }
        return resolvedDeps;
    }

    @Override
    public CacheEntry getBlock(String blockName, BlockCache.Loader loader) {
        Block block;
        Map<String, BlockCache.Loader> deps = loader.getDependencies();
        if (deps.isEmpty()) {
            block = (Block)this.cache.get((Object)blockName, k -> this.load(loader, Collections.emptyMap()));
        } else {
            block = (Block)this.cache.getIfPresent((Object)blockName);
            if (block == null) {
                Map<String, byte[]> resolvedDeps = this.resolveDependencies(deps);
                if (resolvedDeps == null) {
                    return null;
                }
                block = this.cache.asMap().computeIfAbsent(blockName, k -> this.load(loader, resolvedDeps));
            }
        }
        return this.wrap(blockName, block);
    }

    private class TlfuCacheEntry
    implements CacheEntry {
        private final String cacheKey;
        private final Block block;

        TlfuCacheEntry(String k, Block b) {
            this.cacheKey = k;
            this.block = b;
        }

        @Override
        public byte[] getBuffer() {
            return this.block.getBuffer();
        }

        @Override
        public <T extends CacheEntry.Weighable> T getIndex(Supplier<T> supplier) {
            return this.block.getIndex(supplier);
        }

        @Override
        public void indexWeightChanged() {
            if (this.block.indexWeightChanged()) {
                TinyLfuBlockCache.this.cache.put((Object)this.cacheKey, (Object)this.block);
            }
        }
    }

    private static final class Block {
        private final byte[] buffer;
        private CacheEntry.Weighable index;
        private volatile int lastIndexWeight;

        Block(byte[] buffer) {
            this.buffer = buffer;
            this.lastIndexWeight = buffer.length / 100;
        }

        int weight() {
            int indexWeight = this.lastIndexWeight + 4 + ClassSize.REFERENCE;
            return indexWeight + ClassSize.align(this.getBuffer().length) + 8 + ClassSize.REFERENCE + ClassSize.OBJECT + ClassSize.ARRAY;
        }

        public byte[] getBuffer() {
            return this.buffer;
        }

        public synchronized <T extends CacheEntry.Weighable> T getIndex(Supplier<T> supplier) {
            if (this.index == null) {
                this.index = (CacheEntry.Weighable)supplier.get();
            }
            return (T)this.index;
        }

        public synchronized boolean indexWeightChanged() {
            int indexWeight;
            if (this.index != null && (indexWeight = this.index.weight()) > this.lastIndexWeight) {
                this.lastIndexWeight = indexWeight;
                return true;
            }
            return false;
        }
    }
}

