/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.fate.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.security.SecureRandom;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Retry {
    private static final Logger log = LoggerFactory.getLogger(Retry.class);
    private long maxRetries;
    private long waitIncrement;
    private long maxWait;
    private final long logIntervalNanoSec;
    private double backOffFactor;
    private long retriesDone;
    private long currentWait;
    private long initialWait;
    private boolean hasNeverLogged;
    private long lastRetryLog;
    private static Random rand = new SecureRandom();
    private double currentBackOffFactor;
    private boolean doTimeJitter = true;

    private Retry(long maxRetries, long startWait, long waitIncrement, long maxWait, long logInterval, double backOffFactor) {
        this.maxRetries = maxRetries;
        this.maxWait = maxWait;
        this.waitIncrement = waitIncrement;
        this.retriesDone = 0L;
        this.currentWait = startWait;
        this.initialWait = startWait;
        this.logIntervalNanoSec = TimeUnit.MILLISECONDS.toNanos(logInterval);
        this.hasNeverLogged = true;
        this.lastRetryLog = -1L;
        this.currentBackOffFactor = this.backOffFactor = backOffFactor;
    }

    @VisibleForTesting
    public void setBackOffFactor(double baskOffFactor) {
        this.currentBackOffFactor = this.backOffFactor = baskOffFactor;
    }

    @VisibleForTesting
    public double getWaitFactor() {
        return this.backOffFactor;
    }

    @VisibleForTesting
    long getMaxRetries() {
        return this.maxRetries;
    }

    @VisibleForTesting
    long getCurrentWait() {
        return this.currentWait;
    }

    @VisibleForTesting
    long getWaitIncrement() {
        return this.waitIncrement;
    }

    @VisibleForTesting
    long getMaxWait() {
        return this.maxWait;
    }

    @VisibleForTesting
    void setMaxRetries(long maxRetries) {
        this.maxRetries = maxRetries;
    }

    @VisibleForTesting
    void setStartWait(long startWait) {
        this.currentWait = startWait;
        this.initialWait = startWait;
    }

    @VisibleForTesting
    void setWaitIncrement(long waitIncrement) {
        this.waitIncrement = waitIncrement;
    }

    @VisibleForTesting
    void setMaxWait(long maxWait) {
        this.maxWait = maxWait;
    }

    @VisibleForTesting
    void setDoTimeJitter(boolean jitter) {
        this.doTimeJitter = jitter;
    }

    public boolean hasInfiniteRetries() {
        return this.maxRetries < 0L;
    }

    public long getLogInterval() {
        return TimeUnit.NANOSECONDS.toMillis(this.logIntervalNanoSec);
    }

    public boolean canRetry() {
        return this.hasInfiniteRetries() || this.retriesDone < this.maxRetries;
    }

    public void useRetry() {
        if (!this.canRetry()) {
            throw new IllegalStateException("No retries left");
        }
        ++this.retriesDone;
    }

    public boolean hasRetried() {
        return this.retriesDone > 0L;
    }

    public long retriesCompleted() {
        return this.retriesDone;
    }

    public void waitForNextAttempt() throws InterruptedException {
        double waitFactor = (1.0 + (rand.nextDouble() - 0.5) / 10.0) * this.currentBackOffFactor;
        if (!this.doTimeJitter) {
            waitFactor = this.currentBackOffFactor;
        }
        this.currentBackOffFactor *= this.backOffFactor;
        log.debug("Sleeping for {}ms before retrying operation", (Object)this.currentWait);
        this.sleep(this.currentWait);
        if (this.backOffFactor == 1.0) {
            this.currentWait = Math.min(this.maxWait, this.currentWait + this.waitIncrement);
        } else if (this.backOffFactor > 1.0) {
            this.waitIncrement = (long)Math.ceil(waitFactor * (double)this.initialWait);
            this.currentWait = Math.min(this.maxWait, this.initialWait + this.waitIncrement);
        }
    }

    protected void sleep(long wait) throws InterruptedException {
        Thread.sleep(wait);
    }

    public void logRetry(Logger log, String message, Throwable t) {
        long now = System.nanoTime();
        if (this.hasNeverLogged) {
            if (log.isDebugEnabled()) {
                log.debug(this.getMessage(message, t));
            }
            this.hasNeverLogged = false;
            this.lastRetryLog = now;
        } else if (now - this.lastRetryLog > this.logIntervalNanoSec) {
            log.warn(this.getMessage(message), t);
            this.lastRetryLog = now;
        } else if (log.isTraceEnabled()) {
            log.trace(this.getMessage(message, t));
        }
    }

    public void logRetry(Logger log, String message) {
        long now = System.nanoTime();
        if (this.hasNeverLogged) {
            if (log.isDebugEnabled()) {
                log.debug(this.getMessage(message));
            }
            this.hasNeverLogged = false;
            this.lastRetryLog = now;
        } else if (now - this.lastRetryLog > this.logIntervalNanoSec) {
            log.warn(this.getMessage(message));
            this.lastRetryLog = now;
        } else if (log.isTraceEnabled()) {
            log.trace(this.getMessage(message));
        }
    }

    private String getMessage(String message) {
        return message + ", retrying attempt " + (this.retriesDone + 1L) + " (suppressing retry messages for " + this.getLogInterval() + "ms)";
    }

    private String getMessage(String message, Throwable t) {
        return message + ":" + t + ", retrying attempt " + (this.retriesDone + 1L) + " (suppressing retry messages for " + this.getLogInterval() + "ms)";
    }

    public static NeedsRetries builder() {
        return new RetryFactoryBuilder();
    }

    private static class RetryFactoryBuilder
    implements NeedsRetries,
    NeedsRetryDelay,
    NeedsTimeIncrement,
    NeedsMaxWait,
    NeedsLogInterval,
    NeedsBackOffFactor,
    BuilderDone,
    RetryFactory {
        private boolean modifiable = true;
        private long maxRetries;
        private long initialWait;
        private long maxWait;
        private long waitIncrement;
        private long logInterval;
        private double backOffFactor = 1.5;

        RetryFactoryBuilder() {
        }

        private void checkState() {
            Preconditions.checkState((boolean)this.modifiable, (Object)"Cannot modify this builder once 'createFactory()' has been called");
        }

        @Override
        public NeedsRetryDelay infiniteRetries() {
            this.checkState();
            this.maxRetries = -1L;
            return this;
        }

        @Override
        public NeedsRetryDelay maxRetries(long max) {
            this.checkState();
            Preconditions.checkArgument((max >= 0L ? 1 : 0) != 0, (Object)"Maximum number of retries must not be negative");
            this.maxRetries = max;
            return this;
        }

        @Override
        public NeedsTimeIncrement retryAfter(long duration, TimeUnit unit) {
            this.checkState();
            Preconditions.checkArgument((duration >= 0L ? 1 : 0) != 0, (Object)"Initial waiting period must not be negative");
            this.initialWait = unit.toMillis(duration);
            return this;
        }

        @Override
        public NeedsMaxWait incrementBy(long duration, TimeUnit unit) {
            this.checkState();
            Preconditions.checkArgument((duration >= 0L ? 1 : 0) != 0, (Object)"Amount of time to increment the wait between each retry must not be negative");
            this.waitIncrement = unit.toMillis(duration);
            return this;
        }

        @Override
        public NeedsLogInterval backOffFactor(double factor) {
            this.checkState();
            Preconditions.checkArgument((factor >= 1.0 ? 1 : 0) != 0, (Object)"backOffFactor exponent that increases the wait between each retry and must greater than one");
            this.backOffFactor = factor;
            return this;
        }

        @Override
        public NeedsBackOffFactor maxWait(long duration, TimeUnit unit) {
            this.checkState();
            this.maxWait = unit.toMillis(duration);
            Preconditions.checkArgument((this.maxWait >= this.initialWait ? 1 : 0) != 0, (Object)"Maximum wait between retries must not be less than the initial delay");
            return this;
        }

        @Override
        public BuilderDone logInterval(long duration, TimeUnit unit) {
            this.checkState();
            Preconditions.checkArgument((duration >= 0L ? 1 : 0) != 0, (Object)"The amount of time between logging retries must not be negative");
            this.logInterval = unit.toMillis(duration);
            return this;
        }

        @Override
        public RetryFactory createFactory() {
            this.modifiable = false;
            return this;
        }

        @Override
        public Retry createRetry() {
            return new Retry(this.maxRetries, this.initialWait, this.waitIncrement, this.maxWait, this.logInterval, this.backOffFactor);
        }
    }

    public static interface RetryFactory {
        public Retry createRetry();
    }

    public static interface BuilderDone {
        public RetryFactory createFactory();

        public Retry createRetry();
    }

    public static interface NeedsLogInterval {
        public BuilderDone logInterval(long var1, TimeUnit var3);
    }

    public static interface NeedsBackOffFactor {
        public NeedsLogInterval backOffFactor(double var1);
    }

    public static interface NeedsMaxWait {
        public NeedsBackOffFactor maxWait(long var1, TimeUnit var3);
    }

    public static interface NeedsTimeIncrement {
        public NeedsMaxWait incrementBy(long var1, TimeUnit var3);
    }

    public static interface NeedsRetryDelay {
        public NeedsTimeIncrement retryAfter(long var1, TimeUnit var3);
    }

    public static interface NeedsRetries {
        public NeedsRetryDelay infiniteRetries();

        public NeedsRetryDelay maxRetries(long var1);
    }
}

