/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.conf;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import org.apache.accumulo.core.conf.ConfigurationTypeHelper;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.conf.PropertyType;
import org.apache.accumulo.core.spi.crypto.CryptoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigSanityCheck {
    private static final Logger log = LoggerFactory.getLogger(ConfigSanityCheck.class);
    private static final String PREFIX = "BAD CONFIG ";

    public static void validate(Iterable<Map.Entry<String, String>> entries) {
        String instanceZkTimeoutValue = null;
        for (Map.Entry<String, String> entry : entries) {
            String key = entry.getKey();
            String value = entry.getValue();
            Property prop = Property.getPropertyByKey(entry.getKey());
            if (prop == null && Property.isValidPropertyKey(key)) continue;
            if (prop == null) {
                log.warn("BAD CONFIG unrecognized property key (" + key + ")");
            } else if (prop.getType() == PropertyType.PREFIX) {
                ConfigSanityCheck.fatal("BAD CONFIG incomplete property key (" + key + ")");
            } else if (!prop.getType().isValidFormat(value)) {
                ConfigSanityCheck.fatal("BAD CONFIG improperly formatted value for key (" + key + ", type=" + (Object)((Object)prop.getType()) + ") : " + value);
            }
            if (key.equals(Property.INSTANCE_ZK_TIMEOUT.getKey())) {
                instanceZkTimeoutValue = value;
            }
            if (key.equals(Property.TABLE_FILE_COMPRESSED_BLOCK_SIZE_INDEX.getKey()) || key.equals(Property.TABLE_FILE_COMPRESSED_BLOCK_SIZE.getKey())) {
                long bsize = ConfigurationTypeHelper.getFixedMemoryAsBytes(value);
                Preconditions.checkArgument((bsize > 0L && bsize < Integer.MAX_VALUE ? 1 : 0) != 0, (Object)(key + " must be greater than 0 and less than " + Integer.MAX_VALUE + " but was: " + bsize));
            }
            if (!key.equals(Property.INSTANCE_CRYPTO_SERVICE.getKey())) continue;
            String cryptoStrategy = Objects.requireNonNull(value);
            ConfigSanityCheck.verifyValidClassName(key, cryptoStrategy, CryptoService.class);
        }
        if (instanceZkTimeoutValue != null) {
            ConfigSanityCheck.checkTimeDuration(Property.INSTANCE_ZK_TIMEOUT, instanceZkTimeoutValue, new CheckTimeDurationBetween(1000L, 300000L));
        }
    }

    private static void checkTimeDuration(Property prop, String value, CheckTimeDuration chk) {
        ConfigSanityCheck.verifyPropertyTypes(PropertyType.TIMEDURATION, prop);
        if (!chk.check(ConfigurationTypeHelper.getTimeInMillis(value))) {
            ConfigSanityCheck.fatal(PREFIX + chk.getDescription(prop));
        }
    }

    private static void verifyPropertyTypes(PropertyType type, Property ... properties) {
        for (Property prop : properties) {
            if (prop.getType() == type) continue;
            ConfigSanityCheck.fatal("Unexpected property type (" + (Object)((Object)prop.getType()) + " != " + (Object)((Object)type) + ")");
        }
    }

    private static void fatal(String msg) {
        log.error("FATAL: {}", (Object)msg);
        throw new SanityCheckException(msg);
    }

    private static void verifyValidClassName(String confOption, String className, Class<?> requiredBaseClass) {
        try {
            ConfigurationTypeHelper.getClassInstance(null, className, requiredBaseClass);
        }
        catch (IOException | ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            ConfigSanityCheck.fatal(confOption + " has an invalid class name: " + className);
        }
        catch (ClassCastException e) {
            ConfigSanityCheck.fatal(confOption + " must implement " + requiredBaseClass + ", but the configured class does not: " + className);
        }
    }

    public static class SanityCheckException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public SanityCheckException(String msg) {
            super(msg);
        }
    }

    private static class CheckTimeDurationBetween
    implements CheckTimeDuration {
        long min;
        long max;

        CheckTimeDurationBetween(long x, long y) {
            this.min = Math.min(x, y);
            this.max = Math.max(x, y);
        }

        @Override
        public boolean check(long propVal) {
            return propVal >= this.min && propVal <= this.max;
        }

        @Override
        public String getDescription(Property prop) {
            return "ensure " + this.min + " <= " + (Object)((Object)prop) + " <= " + this.max;
        }
    }

    private static interface CheckTimeDuration {
        public boolean check(long var1);

        public String getDescription(Property var1);
    }
}

