% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ml_helper.R
\name{pt_collect_results}
\alias{pt_collect_results}
\title{Collect diagnostics from two \code{ml_backtest_multi()} runs}
\usage{
pt_collect_results(
  bt_pooled_list,
  bt_neutral_list,
  weekly_prices,
  horizons,
  split_date,
  cost_bps = c(5, 10, 15, 20, 25, 50),
  freq = 52,
  prices = NULL,
  ic_roll_window = 26L,
  mask_scores_to_decision_dates = TRUE,
  cost_model = function(turnover, bps) (bps/10000) * turnover
)
}
\arguments{
\item{bt_pooled_list}{Named list of backtests (keys like \code{H4w}, \code{H13w})
produced with \code{group_mode = "pooled"}.}

\item{bt_neutral_list}{Named list of backtests (same keys) produced with
\code{group_mode = "per_group"}.}

\item{weekly_prices}{Deprecated alias for \code{prices}; kept for backwards compatibility.}

\item{horizons}{Integer vector of horizons (in weeks) expected in the lists.}

\item{split_date}{\code{Date} used to split performance into \code{Pre}/\code{Post}.}

\item{cost_bps}{Numeric vector of per-rebalance cost levels (in basis points)
for the turnover-based cost sweep. Default \code{c(5, 10, 15, 20, 25, 50)}.}

\item{freq}{Integer frequency used by \code{perf_metrics()} (e.g., \code{52} for weekly).}

\item{prices}{Optional price table (preferred). If \code{NULL}, \code{weekly_prices} is used.}

\item{ic_roll_window}{Integer window length (weeks) for rolling IC on OOS decision dates.
Default \code{26L}.}

\item{mask_scores_to_decision_dates}{Logical; if \code{TRUE} (default) scores are
masked to OOS decision dates only (see \code{\link[=scores_oos_only]{scores_oos_only()}}).}

\item{cost_model}{Function \verb{(turnover, bps)} returning per-period cost to subtract
from returns in the sweep. Default scales linearly with turnover.}
}
\value{
A named list with one element per horizon, each containing:
\itemize{
\item \code{bt_pooled}, \code{bt_neutral} — the input backtests;
\item \code{coverage} — coverage by date for pooled/neutral;
\item \code{ic_series} — raw IC series for pooled/neutral;
\item \code{icroll_oos_26w} — rolling IC (OOS-only) for pooled/neutral;
\item \code{masked_scores} — OOS-masked score tables for pooled/neutral;
\item \code{perf_tables} — performance tables (Full/Pre/Post);
\item \code{turnover_neutral} — turnover series for the sector-neutral run;
\item \code{cost_sweep_neutral} — performance under gross/net across \code{cost_bps}.
}
}
\description{
Builds a compact set of outputs—coverage, IC series, OOS-only rolling IC,
performance tables (Full/Pre/Post), turnover, and a cost sweep—given two
lists of backtests (pooled and per-group) produced by \code{ml_backtest_multi()}.
}
\details{
Both input lists must have identical horizon keys (\code{paste0("H", h, "w")}).
Coverage and IC series are computed from stored \code{scores}; rolling IC is built
on OOS decision dates only; performance is summarised for the full sample
and \code{Pre}/\code{Post} relative to \code{split_date}; turnover is derived from realised
sector-neutral weights; and a turnover-based cost sweep is evaluated on the
sector-neutral run across \code{cost_bps}.
}
\examples{
\donttest{
if (requireNamespace("PortfolioTesteR", quietly = TRUE)) {
  library(PortfolioTesteR)
  data(sample_prices_weekly, package = "PortfolioTesteR")
  # Simple feature
  mom12 <- PortfolioTesteR::calc_momentum(sample_prices_weekly, 12)
  feats <- list(mom12 = mom12)
  fit_first     <- function(X, y, ...) list()
  predict_first <- function(model, Xnew, ...) as.numeric(Xnew[[1]])
  sch <- list(is = 52L, oos = 26L, step = 26L)
  syms <- setdiff(names(sample_prices_weekly), "Date")
  gmap <- data.frame(Symbol = syms,
                     Group  = rep(c("G1","G2"), length.out = length(syms)))
  bt_pooled  <- ml_backtest_multi(feats, sample_prices_weekly, c(4L),
                                  fit_first, predict_first, sch,
                                  selection = list(top_k = 5L),
                                  weighting = list(method = "softmax", temperature = 12),
                                  caps = list(max_per_symbol = 0.10),
                                  group_mode = "pooled")
  bt_neutral <- ml_backtest_multi(feats, sample_prices_weekly, c(4L),
                                  fit_first, predict_first, sch,
                                  selection = list(top_k = 5L),
                                  weighting = list(method = "softmax", temperature = 12),
                                  caps = list(max_per_symbol = 0.10),
                                  group_mode = "per_group",
                                  group_map = gmap)
  out <- pt_collect_results(
    bt_pooled_list  = bt_pooled,
    bt_neutral_list = bt_neutral,
    prices          = sample_prices_weekly,
    horizons        = c(4L),
    split_date      = as.Date("2019-01-01"),
    cost_bps        = c(5, 15),
    freq            = 52,
    ic_roll_window  = 13L
  )
  names(out)
  str(out[["H4w"]]$perf_tables)
}}

}
\seealso{
\code{\link[=ml_backtest_multi]{ml_backtest_multi()}}, \code{\link[=scores_oos_only]{scores_oos_only()}}, \code{\link[=perf_metrics]{perf_metrics()}}

Other Chapter3-helpers: 
\code{\link{scores_oos_only}()}
}
\concept{Chapter3-helpers}
