% File src/library/grid/man/grid.grab.Rd
% Part of the R package, https://www.R-project.org
% Copyright 1995-2007 R Core Team
% Distributed under GPL 2 or later

\name{grid.grab}
\alias{grid.grab}
\alias{grid.grabExpr}
\title{ Grab the current grid output }
\description{
  Creates a \I{gTree} object from the current grid display list
  or from a scene generated by user-specified code.
}
\usage{
grid.grab(warn = 2, wrap = wrap.grobs, wrap.grobs = FALSE, ...)
grid.grabExpr(expr, warn = 2, wrap = wrap.grobs, wrap.grobs = FALSE, 
              width = 7, height = 7, device = offscreen, ...)
}
\arguments{
  \item{expr}{ An expression to be evaluated.  Typically,
    some calls to grid drawing functions. }
  \item{warn}{ An integer specifying the amount of warnings
    to emit.  0 means no warnings, 1 means warn when it is
    certain that the grab will not faithfully represent the
    original scene. 2 means warn if there's any possibility
    that the grab will not faithfully represent the
    original scene.
  }
  \item{wrap}{ A logical indicating how the output should
    be captured. If \code{TRUE}, each non-grob element on the
    display list is captured by wrapping it in a grob.
  }
  \item{wrap.grobs}{ A logical indicating whether, if we are wrapping
    elements (\code{wrap=TRUE}), we should wrap grobs (or just
    wrap viewports).
  }
  \item{width, height}{ Size of the device used for temporary
    rendering. }
  \item{device}{ A function that opens a graphics device for temporary
    rendering. By default this is an off-screen, in-memory device
    based on the \code{pdf} device, but this default device may not be
    satisfactory when using custom fonts.
  }
  \item{\dots}{ arguments passed to \I{gTree}, for example, a
    name and/or class for the \I{gTree} that is created.}
}
\details{
  There are four ways to capture grid output as a \I{gTree}.

  There are two functions for capturing output:
  use \code{grid.grab} to capture an existing drawing
  and \code{grid.grabExpr} to capture the output from
  an expression (without drawing anything).

  For each of these functions, the output can be captured in
  two ways.  One way tries to be clever and make a
  \I{gTree} with a \I{childrenvp} slot containing all viewports on
  the display list (including those
  that are popped) and every
  grob on the display list as a child of the new
  \I{gTree};  each child has a \I{vpPath} in the \I{vp} slot so that it is
  drawn in the appropriate viewport.
  In other words, the \I{gTree} contains all elements on the display
  list, but in a slightly altered form.

  The other way, \code{wrap=TRUE},
  is to create a grob for every element on the
  display list (and make all of those grobs children of the
  \I{gTree}).  Only viewports are
  wrapped unless \code{wrap.grobs} is also \code{TRUE}.

  The first approach creates a more compact and elegant \I{gTree},
  which is more flexible to work with,
  but is not guaranteed to faithfully replicate all possible
  grid output.  The second approach is more brute force, and
  harder to work with, but is more likely to replicate the original
  output.

  An example of a case that will NOT be replicated by wrapping,
  with \code{wrap.grobs=TRUE}, is
  a scene where the placement of one grob is dependent on another grob
  (e.g., via \code{grobX} or \code{grobWidth}).
}
\value{
  A \I{gTree} object.
}
\seealso{ \code{\link{gTree}}
}
\examples{
pushViewport(viewport(width=.5, height=.5))
grid.rect()
grid.points(stats::runif(10), stats::runif(10))
popViewport()
grab <- grid.grab()
grid.newpage()
grid.draw(grab)
}
\keyword{ dplot }
