% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/marineBackground.R
\name{marineBackground}
\alias{marineBackground}
\title{Marine background shapefile generation}
\usage{
marineBackground(occs, clipToOcean = TRUE, verbose = TRUE, ...)
}
\arguments{
\item{occs}{A \code{data.frame} with at least two columns
named "longitude" and "latitude" or that
can be coerced into this format.}

\item{clipToOcean}{\code{logical}. Clips shapefile to oceans where species
occurs. Useful in cases where buffers jump over narrow
peninsulas (e.g. Isthmus of Panama). Can be quite artificial at ocean
boundaries.}

\item{verbose}{\code{logical}. Switching to \code{FALSE} mutes message describing
which columns in \code{occs} are interpreted as x and y coordinates.}

\item{...}{Additional optional arguments to pass to
\code{getDynamicAlphaHull}.}
}
\value{
A \code{SpatVector}
}
\description{
Automatically generates background
shapefiles for sampling pseudoabsences and/or background
points for niche modeling or species distribution modeling.
Delineating background sampling regions can be one of the
trickiest parts of generating a good model. Automatically
generated background shapefiles should be inspected
carefully prior to model use.

Useful references, among others:
\itemize{
\item Barve N, Barve V, Jiménez-Valverde A, Lira-Noriega A,
Maher SP, Peterson AT, Soberón J, Villalobos F. 2011. The
crucial role of the accessible area in ecological niche
modeling and species distribution modeling.
\emph{Ecological modelling} 222:1810-9.
\item Merow, C, Smith MJ, Silander JA. 2013. A practical
guide to MaxEnt for modeling species’ distributions: what
it does, and why inputs and settings matter." \emph{Ecography}
36: 1058-69.
\item Murphy SJ. 2021. Sampling units derived from geopolitical
boundaries bias biodiversity analyses. \emph{Global Ecology
and Biogeography} 30: 1876-88.
}
}
\details{
The meat of this function is a special-case wrapper
around \code{getDynamicAlphaHull()} from the \code{rangeBuilder} package.
The function documented here is especially useful in cases where
one wants to automatically generate training regions that overlap
the international date line. Regions that exceed the line are cut
and pasted into the appropriate hemisphere instead of being
deleted.

If the argument \code{buff} is not supplied, a buffer is
calculated by taking the mean between the 10th and 90th percentile
of horizontal distances between occurrence points.

If \code{getDynamicAlphaHull()} cannot satisfy the provided conditions,
the occurrences are buffered and then a minimum convex hull is
drawn around the buffer polygons.
}
\examples{
\donttest{
library(terra)
# Create sample raster
r <- rast(ncol=10, nrow=10)
values(r) <- 1:100

# Create test occurrences
set.seed(0)
longitude <- sample(-50:50,
                    size = 20, replace = FALSE)
set.seed(0)
latitude <- sample(-30:30,
                   size = 20, replace = FALSE)
occurrences <- as.data.frame(cbind(longitude,latitude))

# Here's the function
result <- marineBackground(occs = occurrences, buff = 100000,
                           fraction = .9, partCount = 2, clipToOcean = FALSE)
}

}
\seealso{
\code{\link[rangeBuilder:getDynamicAlphaHull]{getDynamicAlphaHull}}
}
\keyword{backgroundSampling}
