% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R, R/temper.R
\docType{package}
\name{temper}
\alias{temper}
\alias{temper-package}
\title{Temporal Encoder–Masked Probabilistic Ensemble Regressor}
\usage{
temper(
  ts,
  future,
  past,
  latent_dim,
  n_trees = 30,
  depth = 6,
  init_prob = 0.8,
  temperature = 0.5,
  n_bases = 10,
  train_rate = 0.7,
  epochs = 30,
  optimizer = "adam",
  lr = 0.005,
  batch = 32,
  lambda_rec = 0.3,
  patience = 15,
  verbose = TRUE,
  alpha = 0.1,
  dates = NULL,
  seed = 42
)
}
\arguments{
\item{ts}{Numeric vector of length at least past + future. Represents the input time series in levels (not log-returns). Missing values are automatically imputed using na_kalman.}

\item{future}{Integer \eqn{\geq 1}. Forecast horizon: the number of steps ahead to predict.}

\item{past}{Integer \eqn{\geq 1}. Length of the sliding window used to feed the encoder.}

\item{latent_dim}{Integer \eqn{\geq 1}. Dimensionality of the autoencoder's latent bottleneck.}

\item{n_trees}{Integer \eqn{\geq 1}. Number of trees in the neural decision forest ensemble. Usually in the range of 30 to 200. Default: 30.}

\item{depth}{Integer \eqn{\geq 1}. Depth of each decision tree (i.e., number of binary splits). Usually in the range of 4 to 12. Default: 6.}

\item{init_prob}{Numeric in \eqn{(0, 1)}. Initial probability that each input feature is kept by the feature mask (used for stochastic feature selection). A value of 0 means always dropped; 1 means always included. Default: 0.8.}

\item{temperature}{Positive numeric. Temperature parameter for the Gumbel–Softmax distribution used during feature masking. Lower values lead to harder (closer to binary) masks; higher values encourage smoother gradients. Default: 0.5.}

\item{n_bases}{Integer \eqn{\geq 1}. Max numbers of bases for the Gaussian mixture. Default: 10.}

\item{train_rate}{Numeric in \eqn{(0, 1)}. Proportion of samples allocated to the training set. The remaining samples form the validation set used for early stopping. Default: 0.7.}

\item{epochs}{Positive integer. Maximum number of training epochs. Have a look at the loss plot to decide the right number of epochs. Default: 30.}

\item{optimizer}{Character string. Optimizer to use for training (adam, adamw, sgd, rprop, rmsprop, adagrad, asgd, adadelta). Default: adam.}

\item{lr}{Positive numeric. Learning rate for the optimizer. Default: 0.005.}

\item{batch}{Positive integer. Mini-batch size used during training. Default: 32.}

\item{lambda_rec}{Non-negative numeric. Weight applied to the reconstruction loss relative to the probabilistic CRPS forecasting loss. Default: 0.3.}

\item{patience}{Positive integer. Number of consecutive epochs without improvement on the validation CRPS before early stopping is triggered. Default: 15.}

\item{verbose}{Logical. If \code{TRUE}, prints CRPS values for each epoch during training. Default: TRUE.}

\item{alpha}{Numeric in \eqn{(0, 1)}. Confidence level used to define the predictive interval band width in the output fan chart. Default: 0.1.}

\item{dates}{Optional \code{Date} vector of the same length as ts. If supplied, fan chart x-axes use calendar dates; otherwise, integer time indices are used. Default: NULL.}

\item{seed}{Optional integer. Used to seed both R and Torch random number generators for reproducibility. Default: 42.}
}
\value{
A named list with four components
\describe{
  \item{`loss`}{A ggplot in which training and validation CRPS are plotted against epoch number, useful for diagnosing over-/under-fitting.}
  \item{`pred_funs`}{A length-`future` list.  Each element contains four empirical distribution functions (pdf, cdf, icdf, sampler) created by empfun}
  \item{`plot`}{A ggplot object showing the historical series, median forecast and predictive interval. A print-ready fan chart.}
  \item{`time_log`}{An object measuring the wall-clock training time.}
}
}
\description{
Temper trains and deploys a hybrid forecasting model that couples a temporal auto-encoder (shrinks a sliding window of length `past` into a latent representation of size `latent_dim`) and a masked neural decision forest (an ensemble of `n_trees` soft decision trees of depth `depth`; feature-level dropout is governed by `init_prob` and annealed by a Gumbel–Softmax with parameter `temperature`) and a CRPS loss (Continuous Ranked Probability Score) that blends the probabilistic forecasting error with a reconstruction term (`lambda_rec × MSE`), to yield multi-step probabilistic forecasts and their fan chart. Model weights are optimized with ADAM or other options, optional early stopping.

Implements a probabilistic ensemble time-series forecaster that combines an auto-encoder with a neural decision forest whose split variables are learned through a differentiable feature-mask layer. Functions are written with 'torch' tensors and provide CRPS (Continuous Ranked Probability Scores) training plus mixture-distribution post-processing.
}
\examples{
\donttest{
set.seed(2025)
ts <- cumsum(rnorm(250))          # synthetic price series
fit <- temper(ts, future = 3, past = 20, latent_dim = 5, epochs = 2)

# 80 \% predictive interval for the 3-step-ahead forecast
pfun <- fit$pred_funs$t3$pfun
pred_interval_80 <- c(pfun(0.1), pfun(0.9))

# Visual diagnostics
print(fit$plot)
print(fit$loss)
}

}
\seealso{
Useful links:
\itemize{
  \item \url{https://rpubs.com/giancarlo_vercellino/temper}
}

}
\author{
\strong{Maintainer}: Giancarlo Vercellino \email{giancarlo.vercellino@gmail.com} [copyright holder]

}
