% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/counterFactuals.R, R/plotMethods.R,
%   R/printMethods.R
\name{cfact_girf}
\alias{cfact_girf}
\alias{plot.cfactgirf}
\alias{print.cfactgirf}
\title{Simulate counterfactual generalized impulse response functions for structural STVAR models.}
\usage{
cfact_girf(
  stvar,
  which_shocks,
  shock_size = 1,
  N = 30,
  R1 = 200,
  R2 = 250,
  init_regime = 1,
  init_values = NULL,
  which_cumulative = numeric(0),
  scale = NULL,
  scale_type = c("instant", "peak"),
  scale_horizon = N,
  ci = c(0.95, 0.8),
  use_data_shocks = FALSE,
  data_girf_pars = c(0, 0.75, 0, 0, 1.5),
  ncores = 2,
  burn_in = 1000,
  exo_weights = NULL,
  seeds = NULL,
  use_parallel = TRUE,
  cfact_type = c("fixed_path", "muted_response"),
  policy_var = 1,
  mute_var = NULL,
  cfact_start = 1,
  cfact_end = 1,
  cfact_path = NULL
)

\method{plot}{cfactgirf}(x, ...)

\method{print}{cfactgirf}(x, ..., digits = 3)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'}, created by, e.g., \code{fitSTVAR} or \code{fitSSTVAR}.}

\item{which_shocks}{a numeric vector of length at most \eqn{d}
(\code{=ncol(data)}) and elements in \eqn{1,...,d} specifying the
structural shocks for which the GIRF should be estimated.}

\item{shock_size}{a non-zero scalar value specifying the common size for all scalar
components of the structural shock. Note that the conditional covariance
matrix of the structural shock is normalized to an identity matrix and that the
(generalized) impulse responses may not be symmetric with respect to the sign
and size of the shock.}

\item{N}{a positive integer specifying the horizon how far ahead should the
generalized impulse responses be calculated.}

\item{R1}{the number of repetitions used to estimate GIRF for each initial
value.}

\item{R2}{the number of initial values to use, i.e., to draw from \code{init_regime}
 if \code{init_values} are not specified. The confidence bounds
 will be sample quantiles of the GIRFs based on different initial values.
 Ignored if the argument \code{init_value} is specified.
@param init_regime an integer in \eqn{1,...,M} specifying the regime from which
 the initial values should be generated from (see \code{?simulate.stvar}). If
 \code{use_data_shocks=TRUE} this is argument not used and \code{data_girf_pars}
 should be specified instead.}

\item{init_regime}{an integer in \eqn{1,...,M} specifying the regime from which
the initial values should be generated from (using a simulation procedure with a burn-in period).
For models with Gaussian conditional distribution, it is also possible to generate the starting
values from the stationary distribution of a regime. See the details section.}

\item{init_values}{a size \code{[p, d, R2]} array specifying the initial values in each slice
for each Monte Carlo repetition, where d is the number of time series in the system and \code{R2}
is an argument of this function. In each slice, the \strong{last} row will be used as initial values
for the first lag, the second last row for second lag etc. If not specified, initial values will be
drawn from the regime specified in \code{init_regimes}.}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the impulse
responses should be cumulative. Default is none.}

\item{scale}{should the GIRFs to some of the shocks be scaled so that they
correspond to a specific magnitude of instantaneous or peak response
of some specific variable (see the argument \code{scale_type})?
Provide a length three vector where the shock of interest
is given in the first element (an integer in \eqn{1,...,d}), the variable of
interest is given in the second element (an integer in \eqn{1,...,d}), and
the magnitude of its instantaneous or peak response in the third element
(a non-zero real number). If the GIRFs of multiple shocks should be scaled, provide
a matrix which has one column for each of the shocks with the columns being
the length three vectors described above.}

\item{scale_type}{If argument \code{scale} is specified, should the GIRFs be
scaled to match an instantaneous response (\code{"instant"}) or peak response
(\code{"peak"}). If \code{"peak"}, the scale is based on the largest magnitude
of peak response in absolute value. Ignored if \code{scale} is not specified.}

\item{scale_horizon}{If \code{scale_type == "peak"} what the maximum horizon up
to which peak response is expected? Scaling won't based on values after this.}

\item{ci}{a numeric vector with elements in \eqn{(0, 1)} specifying the
confidence levels of the "confidence intervals" that \strong{do not} quantify uncertainty about the true parameter value
but only uncertainty about the initial value (and possibly sign and size of the shock) within the given regime.}

\item{use_data_shocks}{set \code{TRUE} for a special feature in which for every possible length \eqn{p} history in the data,
or a subset of them if so specified in the argument \code{data_girf_pars}, the GIRF is estimated for a shock that has the
sign and size of the corresponding structural shock recovered from the data. If used, the argument \code{which_shocks}
must specify only one shock. See the details section.}

\item{data_girf_pars}{a length five numeric vector with the following elements determining settings for \code{use_data_shocks=TRUE}
(concerns the single shock specified in the argument \code{which_shocks}):
\enumerate{
  \item An integer between \code{0} and \code{M} determining the (dominant) regime for which the GIRF should be calculated (\code{0}
    for all regimes).
  \item A number between \code{0.5} and \code{1} determining how large transition weight a regime should have to be considered dominant
    in a given time period (i.e., determining which histories are used to calculate the GIRF if the first element is not \code{0}).
  \item Either \code{0}, \code{-1}, or \code{1}, determining whether the GIRF should be calculated using shocks of all signs,
    only negative shocks, or only positive shocks, respectively.
  \item Either, \code{0}, \code{1}, or \code{2}, determining whether the GIRF should be calculated using shocks of all sizes,
    only small shocks, or only large shocks, respectively.
  \item A strictly positive real number determining what size shocks are considered large and what size small "in the scale of standard
    deviations" (for example, if set to \code{2}, shocks larger than that are considered large and shocks smaller than that are considered
    small; note that the standard deviations of the shocks are normalized to unity).
}}

\item{ncores}{the number CPU cores to be used in parallel computing. Only
single core computing is supported if an initial value is specified (and
the GIRF won't thus be estimated multiple times).}

\item{burn_in}{Burn-in period for simulating initial values from a regime.}

\item{exo_weights}{if \code{weight_function="exogenous"}, provide a size
\eqn{(N+1 \times M)} matrix of exogenous transition weights for the regimes: \code{[h, m]}
for the (after-the-impact) period \eqn{h-1} and regime \eqn{m} weight (\code{[1, m]}
is for the impact period). Ignored if \code{weight_function != "exogenous"}.}

\item{seeds}{A numeric vector initializing the seeds for the random number generator
for estimation of each GIRF. Should have the length of at least (extra seeds are removed
from the end of the vector)...
 \describe{
   \item{If initial values are drawn using \code{init_regime}:}{\code{R2}}
   \item{If initial values are specified in \code{init_values}:}{\code{dim(init_values)[3]}}
   \item{If \code{use_data_shocks=TRUE}:}{1 (the vector of seeds are generated according on the number of histories
     in the data that satisfy the conditions given in the argument \code{data_girf_pars}).}
 }
 Set \code{NULL} for not initializing the seed.}

\item{use_parallel}{employ parallel computing? If \code{FALSE}, does not print
anything.}

\item{cfact_type}{a character string indicating the type of counterfactual to be computed: should the path of the policy
variable be fixed to some hypothetical path (\code{cfact_type="fixed_path"}) in given impulse response horizons or should the responses
of the policy variable to lagged and contemporaneous movements of some given variable be muted (\code{cfact_type="muted_response"})?
See details for more information.}

\item{policy_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the policy variable considered in the
counterfactual scenario. Note that \code{policy_var} is assumed to satisfy \code{!(policy_var \%in\% which_shocks)}.}

\item{mute_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the variable to whose movements the policy variable
specified in the argument \code{policy_var} should not react to in the counterfactual scenario. This indicates also the index of the shock
to which the policy variable should not react to. It is assumed that \code{mute_var != policy_var}. This argument is only used when
\code{cfact_type="muted_response"}.}

\item{cfact_start}{a positive integer between \eqn{0} and \code{N} indicating the starting impulse response horizon period for the
counterfactual behavior of the specified policy variable.}

\item{cfact_end}{a positive integer between \code{cfact_start} and \code{N} indicating the ending period for the counterfactual
behavior of the specified policy variable.}

\item{cfact_path}{a numeric vector of length \code{cfact_end-cfact_start+1} indicating the hypothetical path of the policy variable
specified in the argument \code{policy_var}. This argument is only used when \code{cfact_type="fixed_path"}.}

\item{x}{object of class \code{'cfactgirf'} created by the function \code{cfact_girf}.}

\item{...}{parameters passed to \code{print.stvargirf} printing the girf.}

\item{digits}{how many significant digits to print?}
}
\value{
Returns a class \code{'cfactgirf'} list with the following elements:
  \describe{
    \item{\code{$girf}}{An object of class \code{'girf'} containing the counterfactual GIRFs (see \code{?GIRF}).}
    \item{\code{$stvar}}{The original STVAR model object.}
    \item{\code{$input}}{A list containing the arguments used to calculate the counterfactual.}
 }

Returns the input object \code{x} invisibly.
}
\description{
\code{cfact_girf} simulates counterfactual generalized impulse response functions for structural STVAR models.
}
\details{
Two types of counterfactual generalized impulse response functions (GIRFs) are accommodated where in given impulse response
 horizons either (1) the policy variable of interest takes some hypothetical path (\code{cfact_type="fixed_path"}), or (2)
 its responses to lagged and contemporaneous movements of some given variable are shut off (\code{cfact_type="muted_response"}).
 In both cases, the counterfactual scenarios are simulated by creating hypothetical shocks to the policy variable of interest
 that yield the counterfactual outcome. This approach has the appealing feature that the counterfactual deviations from the
 policy reaction function are treated as policy surprises, allowing them to propagate normally, so that the dynamics of the model
 are not, per se, tampered but just the policy surprises are.

 \strong{Important:} This function assumes that when the policy variable of interest is the \eqn{i_1}th variable, the shock
 to it that is manipulated is the \eqn{i_1}th shock. This should be automatically satisfied for recursively identified models,
 whereas for model identified by heteroskedasticity or non-Gaussianity, the ordering of the shocks can be generally changed
 without loss of generality with the function \code{reorder_B_columns}. In Type (2) counterfactuals it is additionally assumed
 that, if the variable to whose movements the policy variable should not react to is the \eqn{i_2}th variable, the shock to it
 is the \eqn{i_2}th shock. If it is not clear whether the \eqn{i_2}th shock can be interpreted as a shock to a variable
 (but has a broader definition such as "a demand shock"), the Type (2) counterfactual scenario is interpreted as follows: the \eqn{i_1}th
 variable does not react to lagged movements of the \eqn{i_2}th variable nor to the \eqn{i_2}th shock.

 See the seminal paper of Bernanke et al (1997) for discussing about the "Type (1)" counterfactuals and
 Kilian and Lewis (2011) for discussion about the "Type (2)" counterfactuals. See Kilian and Lütkepohl (2017), Section 4.5
 for further discussion about counterfactuals. The literature cited about considers linear models, but it is
 explained in the vignette of this package how this function computes the historical counterfactuals for the STVAR models in
 a way that accommodates nonlinear time-varying dynamics.
}
\section{Functions}{
\itemize{
\item \code{plot(cfactgirf)}: plot method

\item \code{print(cfactgirf)}: print method

}}
\examples{
\donttest{
# Recursively identified logistic Student's t STVAR(p=3, M=2) model with the first
# lag of the second variable as the switching variable:
params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
  0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

# Counterfactual GIRFs for Shock 2 with horizon N=5 (using only R1=R2=10 Monte Carlo repetitions
# to save computation time), where the first variable takes values 1, -2, and 3 in the
# horizons 1, 2, and 3, respectively:
cfact1 <- cfact_girf(mod32logt, which_shocks=2, N=5, R1=10, R2=10, init_regime=1, seeds=1:10,
 cfact_type="fixed_path", policy_var=1, cfact_start=1, cfact_end=3, cfact_path=c(1, -2, 3))
cfact1 # Print the results
plot(cfact1) # Plot the counterfactual GIRF

# Counterfactual GIRFs for Shock 2 with horizon N=5 (using only R1=R2=10 Monte Carlo repetitions
# to save computation time), where the first variable does not respond to lagged movements
# of the second variable nor to the second shock in time periods from 1 to 3:
cfact2 <- cfact_girf(mod32logt, which_shocks=2, N=5, R1=10, R2=10, init_regime=1, seeds=1:20,
 cfact_type="muted_response", policy_var=1, mute_var=2, cfact_start=1, cfact_end=3)
cfact2 # Print the results
plot(cfact2) # Plot the counterfactual GIRF
}
}
\references{
\itemize{
   \item Bernanke B., Gertler M., Watson M. 1997. Systematic monetary policy and the effects of oilprice shocks.
     \emph{Brookings Papers on Economic Activity}, 1, 91—142.
   \item Kilian L., Lewis L. 2011. Does the fed respond to oil price shocks? \emph{The Economic Journal}, \strong{121}:555.
   \item Kilian L., Lütkepohl H. 2017. Structural Vector Autoregressive Analysis. 1st edition.
     \emph{Cambridge University Press}, Cambridge.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{linear_IRF}}, \code{\link{hist_decomp}}, \code{\link{cfact_hist}},
 \code{\link{cfact_fore}},  \code{\link{fitSSTVAR}}
}
