% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparseGFM.R
\name{cv.sparseGFM}
\alias{cv.sparseGFM}
\title{Cross-Validation for Sparse Generalized Factor Model}
\usage{
cv.sparseGFM(
  x,
  type = c("continuous", "count", "binary"),
  q = 2,
  penalty = c("lasso", "SCAD", "MCP", "group", "agroup", "gSCAD", "agSCAD", "gMCP",
    "agMCP", "alasso", "glasso", "aglasso"),
  lambda_range = seq(0.1, 1, by = 0.1),
  gam = 1,
  tau = NULL,
  mat_sd = 1,
  delta = 1e-04,
  maxiter = 30,
  C = 5,
  bic_type = "dd",
  verbose = TRUE
)
}
\arguments{
\item{x}{A numeric matrix of observations (n x p)}

\item{type}{Character string specifying the data type ("continuous", "count", or "binary")}

\item{q}{Integer specifying the number of latent factors (default = 2)}

\item{penalty}{Character string specifying the penalty type. See sparseGFM for all 12 available options.
Recommended: "aglasso" for row-sparse loading matrices}

\item{lambda_range}{Numeric vector of lambda values to evaluate (default = seq(0.1, 1, by = 0.1))}

\item{gam}{Numeric value for the adaptive weight parameter (default = 1)}

\item{tau}{Numeric value for the shape parameter in SCAD/MCP penalties}

\item{mat_sd}{Standard deviation for continuous data (default = 1)}

\item{delta}{Convergence tolerance (default = 1e-4)}

\item{maxiter}{Maximum number of iterations (default = 30)}

\item{C}{Constraint bound for stability (default = 5)}

\item{bic_type}{Character string specifying BIC type. Options are:
\itemize{
  \item "dd": Default SIC using estimated degrees of freedom
  \item "as": Alternative simplified BIC
}}

\item{verbose}{Logical indicating whether to print progress (default = TRUE)}
}
\value{
A list containing:
\itemize{
  \item optimal_lambda: Selected optimal lambda value
  \item optimal_model: Model fitted with optimal lambda
  \item all_results: List of all fitted models for each lambda
  \item objloglik: Vector of log-likelihood values for each lambda
  \item bic_dd: Vector of default SIC values
  \item bic_as: Vector of alternative SIC values
  \item df_dd: Vector of degrees of freedom (default method)
  \item df_as: Vector of degrees of freedom (alternative method)
  \item lambda_range: Vector of evaluated lambda values
}
}
\description{
Performs cross-validation to select the optimal lambda parameter for sparse generalized
factor models using SIC (Sparsity Information Criterion). The method can handle weak
factor scenarios and is particularly effective for row-sparse loading structures.
}
\details{
The function fits sparse GFM models for each lambda value in lambda_range and
calculates two types of SIC for model selection. The optimal lambda is chosen
as the one minimizing the selected SIC criterion. The method automatically handles
missing values and adapts to weak factor structures.
}
\examples{
\donttest{
# Generate data with sparse loading structure
library(sparseGFM)
set.seed(123)
n <- 200; p <- 200; q <- 2
a_param <- 0.9; s <- ceiling(p^a_param)

FF <- matrix(runif(n * q, min = -3, max = 3), nrow = n, ncol = q)
BB <- rbind(matrix(runif(s * q, min = 1, max = 2), nrow = s, ncol = q),
            matrix(0, nrow = (p - s), ncol = q))
alpha_true <- runif(p, min = -1, max = 1)

ident_res <- add_identifiability(FF, BB, alpha_true)
FF0 <- ident_res$H; BB0 <- ident_res$B; alpha0 <- ident_res$mu

mat_para <- FF0 \%*\% t(BB0) + as.matrix(rep(1, n)) \%*\% t(as.matrix(alpha0))
x <- matrix(nrow = n, ncol = p)
for (i in 1:n) {
  for (j in 1:p) {
    x[i, j] <- rnorm(1, mean = mat_para[i, j])
  }
}

# Cross-validation for optimal lambda selection
cv_result <- cv.sparseGFM(x, type = "continuous", q = 2,
                          penalty = "aglasso", C = 5,
                          lambda_range = seq(0.1, 1, by = 0.1),
                          verbose = FALSE)

print(paste("Optimal lambda:", cv_result$optimal_lambda))
optimal_model <- cv_result$optimal_model
}

}
