% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elfunctions.R
\name{EL0}
\alias{EL0}
\title{Uni-variate empirical likelihood via direct lambda search}
\usage{
EL0(
  z,
  mu = NULL,
  ct = NULL,
  shift = NULL,
  return.weights = FALSE,
  SEL = FALSE,
  weight.tolerance = NULL,
  boundary.tolerance = 1e-09,
  trunc.to = 0,
  chull.fail = c("taylor", "wald", "adjusted", "balanced", "none"),
  uniroot.control = list(),
  verbose = FALSE
)
}
\arguments{
\item{z}{Numeric data vector.}

\item{mu}{Hypothesized mean of \code{z} in the moment condition.}

\item{ct}{Numeric count variable with non-negative values that indicates the multiplicity of observations.
Can be fractional. Very small counts below the threshold \code{weight.tolerance} are zeroed.}

\item{shift}{The value to add in the denominator (useful in case there are extra Lagrange multipliers): 1 + lambda'Z + shift.}

\item{return.weights}{Logical: if TRUE, individual EL weights are computed and returned.
Setting this to FALSE gives huge memory savings in large data sets, especially when smoothing is used.}

\item{SEL}{If \code{FALSE}, then the boundaries for the lambda search are based on the total sum of counts, like in vanilla empirical likelihood,
due to formula (2.9) in \insertCite{owen2001empirical}{smoothemplik}, otherwise according to Cosma et al. (2019, p. 170, the topmost formula).}

\item{weight.tolerance}{Weight tolerance for counts to improve numerical stability
(similar to the ones in Art B. Owen's 2017 code, but adapting to the sample size).}

\item{boundary.tolerance}{Relative tolerance for determining when the lambda is not an interior
solution because it is too close to the boundary. Corresponds to a fraction of the
interval range length.}

\item{trunc.to}{Counts under \code{weight.tolerance} will be set to this value.
In most cases, setting this to \code{0} or \code{weight.tolerance} is a viable solution of the zero-denominator problem.}

\item{chull.fail}{A character: what to do if the convex hull of \code{z} does not contain \code{mu}
(spanning condition does not hold). \code{"taylor"} creates a Taylor approximation
of the log-ELR function near the ends of the sample. \code{"wald"} smoothly transitions
between the log-ELR function into -0.5 * the Wald statistic for the weighted mean of \code{z}.
\code{"adjusted"} invokes the method of \insertCite{chen2008adjusted}{smoothemplik},
where an extra observation is added to ensure that the convex hull contains the mean, and
\code{"balanced"} calls the method of \insertCite{emerson2009calibration}{smoothemplik}
and \insertCite{liu2010adjusted}{smoothemplik} with two extra points.}

\item{uniroot.control}{A list passed to the \code{brentZero}.}

\item{verbose}{Logical: if \code{TRUE}, prints warnings.}
}
\value{
A list with the following elements:

\describe{
  \item{logelr}{Logarithm of the empirical likelihood ratio.}
  \item{lam}{The Lagrange multiplier.}
  \item{wts}{Observation weights/probabilities (of the same length as \code{z}).}
  \item{converged}{\code{TRUE} if the algorithm converged, \code{FALSE} otherwise (usually means that \code{mu} is not within the range of \code{z}, i.e. the one-dimensional convex hull of \code{z}).}
  \item{iter}{The number of iterations used (from \code{brentZero}).}
  \item{bracket}{The admissible interval for lambda (that is, yielding weights between 0 and 1).}
  \item{estim.prec}{The approximate estimated precision of lambda (from \code{brentZero}).}
  \item{f.root}{The value of the derivative of the objective function w.r.t. lambda at the root (from \code{brentZero}). Values \code{> sqrt(.Machine$double.eps)} indicate convergence problems.}
  \item{exitcode}{An integer indicating the reason of termination.}
  \item{message}{Character string describing the optimisation termination status.}
}
}
\description{
Empirical likelihood with counts to solve one-dimensional problems efficiently with Brent's root search algorithm.
Conducts an empirical likelihood ratio test of the hypothesis that the mean of \code{z} is \code{mu}.
The names of the elements in the returned list are consistent with the original R code in \insertCite{owen2017weighted}{smoothemplik}.
}
\details{
This function provides the core functionality for univariate empirical likelihood.
The technical details is given in \insertCite{cosma2019inference}{smoothemplik},
although the algorithm used in that paper is slower than the one provided by this function.

Since we know that the EL probabilities belong to (0, 1), the interval (bracket) for \eqn{\lambda}{l} search
can be determined in the spirit of formula (2.9) from \insertCite{owen2001empirical}{smoothemplik}. Let
\eqn{z^*_i := z_i - \mu}{Z[i] := z[i] - mu} be the recentred observations.
\deqn{p_i = c_i/N \cdot (1 + \lambda z^*_i + s)^{-1}}{p[i] = c[i]/N * 1/(1 + l*Z[i] + s)}
The probabilities are bounded from above: \eqn{p_i<1}{p[i] < 1} for all \emph{i}, therefore,
\deqn{c_i/N \cdot (1 + \lambda z^*_i + s)^{-1} < 1}{c[i]/N * 1/(1 + l*Z[i] + s) < 1}
\deqn{c_i/N - 1 - s < \lambda z^*_i}{c[i]/N - 1 - s < l*Z[i]}
Two cases: either \eqn{z^*_i<0}{Z[i] < 0}, or \eqn{z^*_i>0}{Z[i] > 0}
(cases with \eqn{z^*_i=0}{Z[i] = 0} are trivially excluded because they do not affect the EL). Then,
\deqn{(c_i/N - 1 - s)/z^*_i > \lambda,\ \forall i: z^*_i<0}{(c[i]/N - 1 - s)/Z[i] > l,  such i that Z[i]<0}
\deqn{(c_i/N - 1 - s)/z^*_i < \lambda,\ \forall i: z^*_i>0}{(c[i]/N - 1 - s)/Z[i] < l,  such i that Z[i]>0}
which defines the search bracket:
\deqn{\lambda_{\min} := \max_{i: z^*_i>0} (c_i/N - 1 - s)/z^*_i}{l > max_{i: Z[i]>0} (c_i/N - 1 - s)/Z[i]}
\deqn{\lambda_{\max} := \min_{i: z^*_i<0} (c_i/N - 1 - s)/z^*_i}{l < min_{i: Z[i]<0} (c_i/N - 1 - s)/Z[i]}
\deqn{\lambda_{\min} < \lambda < \lambda_{\max}}

(This derivation contains \emph{s}, which is the extra shift that extends the
function to allow mixed conditional and unconditional estimation;
Owen's textbook formula corresponds to \eqn{s = 0}{s = 0}.)

The actual tolerance of the lambda search in \code{brentZero} is
\eqn{2 |\lambda_{\max}| \epsilon_m + \mathrm{tol}/2}{2 * MachEps * l_max + tol/2},
where \code{tol} can be set in \code{uniroot.control} and
\eqn{\epsilon_m}{MachEps} is \code{.Machine$double.eps}.

The sum of log-weights is maximised without Taylor expansion, forcing \code{mu} to be inside
the convex hull of \code{z}. If a violation is happening, consider using the \code{chull.fail} argument
or switching to Euclidean likelihood via [EuL()].
}
\examples{
# Figure 2.4 from Owen (2001) -- with a slightly different data point
earth <- c(
  5.5, 5.61, 4.88, 5.07, 5.26, 5.55, 5.36, 5.29, 5.58, 5.65, 5.57, 5.53, 5.62, 5.29,
  5.44, 5.34, 5.79, 5.1, 5.27, 5.39, 5.42, 5.47, 5.63, 5.34, 5.46, 5.3, 5.75, 5.68, 5.85
)
set.seed(1)
system.time(r1 <- replicate(40, EL(sample(earth, replace = TRUE), mu = 5.517)))
set.seed(1)
system.time(r2 <- replicate(40, EL0(sample(earth, replace = TRUE), mu = 5.517)))
plot(apply(r1, 2, "[[", "logelr"), apply(r1, 2, "[[", "logelr") - apply(r2, 2, "[[", "logelr"),
     bty = "n", xlab = "log(ELR) computed via dampened Newthon method",
     main = "Discrepancy between EL and EL0", ylab = "")
abline(h = 0, lty = 2)

# Handling the convex hull violation differently
EL0(1:9, chull.fail = "none")
EL0(1:9, chull.fail = "taylor")
EL0(1:9, chull.fail = "wald")

# Interpolation to well-defined branches outside the convex hull
mu.seq <- seq(-1, 7, 0.1)
wEL1 <- -2*sapply(mu.seq, function(m) EL0(1:9, mu = m, chull.fail = "none")$logelr)
wEL2 <- -2*sapply(mu.seq, function(m) EL0(1:9, mu = m, chull.fail = "taylor")$logelr)
wEL3 <- -2*sapply(mu.seq, function(m) EL0(1:9, mu = m, chull.fail = "wald")$logelr)
plot(mu.seq, wEL1)
lines(mu.seq, wEL2, col = 2)
lines(mu.seq, wEL3, col = 4)

# Warning: depending on the compiler, the discrepancy between EL and EL0
# can be one million (1) times larger than the machine epsilon despite both of them
# being written in pure R
# The results from Apple clang-1400.0.29.202 and Fortran GCC 12.2.0 are different from
# those obtained under Ubuntu 22.04.4 + GCC 11.4.0-1ubuntu1~22.04,
# Arch Linux 6.6.21 + GCC 14.1.1, and Windows Server 2022 + GCC 13.2.0
out1 <- EL(earth, mu = 5.517)[1:4]
out2 <- EL0(earth, mu = 5.517, return.weights = TRUE)[1:4]
print(c(out1$lam, out2$lam), 16)

# Value of lambda                                 EL                  EL0
# aarch64-apple-darwin20         -1.5631313955??????   -1.5631313957?????
# Windows, Ubuntu, Arch           -1.563131395492627   -1.563131395492627
}
\references{
\insertAllCited{}
}
\seealso{
[EL()]
}
