#' Load Sample Data for Package Examples
#' 
#' Loads and subsets sample data stored in the internal datasets 
#' \code{sampleMap} and \code{sampleTimeSeries}, included with the package. 
#' Intended for use in documentation examples instead of \code{Load()}, in order
#' to avoid reliance on external system dependencies like 'nco' and 'cdo' and 
#' to reduce computation time during CRAN checks.
#'
#' @param sdates Character vector of start dates to include (e.g., 
#'  \code{c("19851101", "19901101")}). Must match one or more of the available 
#'  dates in the internal sample datasets: "19851101", "19901101", "19951101", 
#'  "20001101" or "20051101".
#' @param leadtimemin Integer specifying the first lead time to include. Default
#'  is 1.
#' @param leadtimemax Integer specifying the last lead time to include. If NULL, 
#'  the full lead time range is included. Default is NULL.
#' @param output Character string indicating the type of output: \code{"areave"}
#'  for area-averaged time series data, or \code{"lonlat"} for gridded spatial 
#'  data. Default is \code{"areave"}.
#'
#' @return A named list with the following elements:
#' \item{mod}{Array of model data with the selected start dates and lead times.}
#' \item{obs}{Array of observational data with the selected start dates and lead
#'  times.}
#' \item{lat}{Vector of latitudes.}
#' \item{lon}{Vector of longitudes.}
#'
#' @details
#' This function is designed for use in examples and tests within the 
#' 's2dv' package. It provides quick access to precomputed datasets, helping to 
#' avoid slow or system-dependent operations during automated checks.
#'
#' @examples
#' startDates <- c("19851101", "19901101", "19951101", "20001101", "20051101")
#' sampleData <- LoadSampleData(sdates = startDates, output = "areave")
#'
#' @export
LoadSampleData <- function(sdates, leadtimemin = 1, leadtimemax = NULL, 
                            output = 'areave') {
  
  selected_start_dates <- match(sdates, c('19851101', '19901101', '19951101', 
                                          '20001101', '20051101'))
  start_dates_position <- 3
  lead_times_position <- 4

  if (output == 'lonlat') {
    sampleData <- s2dv::sampleMap
    if (is.null(leadtimemax)) {
      leadtimemax <- dim(sampleData$mod)[lead_times_position]
    }
    selected_lead_times <- leadtimemin:leadtimemax

    dataOut <- sampleData
    dataOut$mod <- sampleData$mod[, , selected_start_dates, selected_lead_times, , ]
    dataOut$obs <- sampleData$obs[, , selected_start_dates, selected_lead_times, , ]
  } else if (output == 'areave') {
    sampleData <- s2dv::sampleTimeSeries
    if (is.null(leadtimemax)) {
      leadtimemax <- dim(sampleData$mod)[lead_times_position]
    }
    selected_lead_times <- leadtimemin:leadtimemax

    dataOut <- sampleData
    dataOut$mod <- sampleData$mod[, , selected_start_dates, selected_lead_times]
    dataOut$obs <- sampleData$obs[, , selected_start_dates, selected_lead_times]
  }

  dims_out <- dim(sampleData$mod)
  dims_out[start_dates_position] <- length(selected_start_dates)
  dims_out[lead_times_position] <- length(selected_lead_times)
  dim(dataOut$mod) <- dims_out

  dims_out <- dim(sampleData$obs)
  dims_out[start_dates_position] <- length(selected_start_dates)
  dims_out[lead_times_position] <- length(selected_lead_times)
  dim(dataOut$obs) <- dims_out

  return(list(mod = dataOut$mod, obs = dataOut$obs, 
              lat = dataOut$lat, lon = dataOut$lon))
}
