% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/doubly_balanced_sampling.R
\name{Doubly balanced sampling}
\alias{Doubly balanced sampling}
\alias{local_cube}
\alias{local_cube_stratified}
\title{Doubly balanced sampling}
\usage{
local_cube(probabilities, spread_mat, balance_mat, ...)

local_cube_stratified(probabilities, spread_mat, balance_mat, strata, ...)
}
\arguments{
\item{probabilities}{A vector of inclusion probabilities.}

\item{spread_mat}{A matrix of spreading covariates.}

\item{balance_mat}{A matrix of balancing covariates.}

\item{...}{
  Arguments passed on to \code{\link[=.sampling_defaults]{.sampling_defaults}}
  \describe{
    \item{\code{eps}}{A small value used when comparing floats.}
    \item{\code{bucket_size}}{The maximum size of the k-d-tree nodes. A higher value gives a slower
k-d-tree, but is faster to create and takes up less memory.}
  }}

\item{strata}{An integer vector with stratum numbers for each unit.}
}
\value{
A vector of sample indices.
}
\description{
Selects doubly balanced samples with prescribed inclusion probabilities from finite populations.
}
\details{
For the \code{local_cube} method, a fixed sized sample is obtained if the first column of
\code{balance_mat} is the inclusion probabilities. For \code{local_cube_stratified}, the inclusion
probabilities are inserted automatically.
}
\section{Functions}{
\itemize{
\item \code{local_cube()}: The local cube method

\item \code{local_cube_stratified()}: The stratified local cube method

}}
\examples{
set.seed(12345);
N = 1000;
n = 100;
prob = rep(n/N, N);
xb = matrix(c(prob, runif(N * 2)), ncol = 3);
xs = matrix(runif(N * 2), ncol = 2);
strata = c(rep(1L, 100), rep(2L, 200), rep(3L, 300), rep(4L, 400));

s = local_cube(prob, xs, xb);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = local_cube_stratified(prob, xs, xb[, -1], strata);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

\donttest{
# Respects inclusion probabilities
set.seed(12345);
prob = c(0.2, 0.25, 0.35, 0.4, 0.5, 0.5, 0.55, 0.65, 0.7, 0.9);
N = length(prob);
xb = matrix(c(prob, runif(N * 2)), ncol = 3);
xs = matrix(runif(N * 2), ncol = 2);

ep = rep(0L, N);
r = 10000L;

for (i in seq_len(r)) {
  s = local_cube(prob, xs, xb);
  ep[s] = ep[s] + 1L;
}

print(ep / r - prob);
}

}
\references{
Deville, J. C. and Tillé, Y. (2004).
Efficient balanced sampling: the cube method.
Biometrika, 91(4), 893-912.

Chauvet, G. and Tillé, Y. (2006).
A fast algorithm for balanced sampling.
Computational Statistics, 21(1), 53-62.

Chauvet, G. (2009).
Stratified balanced sampling.
Survey Methodology, 35, 115-119.

Grafström, A. and Tillé, Y. (2013).
Doubly balanced spatial sampling with spreading and restitution of auxiliary totals.
Environmetrics, 24(2), 120-131
}
