% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vect_link_ng.R, R/vect_link.R
\name{read_VECT}
\alias{read_VECT}
\alias{write_VECT}
\alias{vInfo}
\alias{vColumns}
\alias{vDataCount}
\alias{vect2neigh}
\title{Read and write GRASS vector object files}
\usage{
read_VECT(
  vname,
  layer = "",
  proxy = FALSE,
  use_gdal_grass_driver = TRUE,
  type = NULL,
  flags = "overwrite",
  Sys_ignore.stdout = FALSE,
  ignore.stderr = get.ignore.stderrOption()
)

write_VECT(
  x,
  vname,
  flags = "overwrite",
  ignore.stderr = get.ignore.stderrOption()
)

vInfo(vname, layer, ignore.stderr = NULL)

vColumns(vname, layer, ignore.stderr = NULL)

vDataCount(vname, layer, ignore.stderr = NULL)

vect2neigh(
  vname,
  ID = NULL,
  ignore.stderr = NULL,
  remove = TRUE,
  vname2 = NULL,
  units = "k"
)
}
\arguments{
\item{vname}{A GRASS vector file name.}

\item{layer}{a layer name (string); if missing the first layer will be used.}

\item{proxy}{Default is \code{FALSE}. Set as \code{TRUE} if you need a
\code{SpatVectorProxy} object.}

\item{use_gdal_grass_driver}{Default \code{TRUE}. The
\href{https://github.com/OSGeo/gdal-grass}{standalone GDAL-GRASS driver}
for the vector format will be used if it is installed. The advantage is
that no intermediate file needs to be written from GRASS GIS and
subsequently read into R; instead the vector layer is read directly from
the GRASS GIS database. Please read the \strong{Note} further below!.}

\item{type}{override type detection when multiple types are non-zero, passed
to v.out.ogr.}

\item{flags}{Character vector containing additional optional flags and/or
options for v.in.ogr, particularly "o" and "overwrite".}

\item{Sys_ignore.stdout}{Passed to \code{system}.}

\item{ignore.stderr}{default the value set by \code{set.ignore.stderrOption};
NULL, taking the value set by \code{set.ignore.stderrOption}, can be set to
TRUE to silence \code{system()} output to standard error; does not apply on
Windows platforms.}

\item{x}{A \code{SpatVector} object moved to GRASS.}

\item{ID}{A valid DB column name for unique identifiers (optional).}

\item{remove}{default TRUE, remove copied vectors created in
\code{vect2neigh}.}

\item{vname2}{If on a previous run, remove was FALSE, the name of the
temporary vector may be given to circumvent its generation.}

\item{units}{default "k"; see GRASS 'v.to.db' manual page for alternatives.}
}
\value{
\code{read_VECT} imports a GRASS vector layer into a \code{SpatVector} or
\code{SpatVectorProxy} object.
\code{vect2neigh} returns a data frame object with left and right
neighbours and boundary lengths, also given class GRASSneigh and
spatial.neighbour (as used in spdep). The incantation to retrieve the
neighbours list is \code{sn2listw(vect2neigh())$neighbours}, and to
retrieve the boundary lengths: \code{sn2listw(vect2neigh())$weights}. The
GRASSneigh object has two other useful attributes: external is a vector
giving the length of shared boundary between each polygon and the external
area, and total giving each polygon's total boundary length.
}
\description{
\code{read_VECT} moves one GRASS vector object file with attribute data
through a temporary GeoPackage file to a \code{\link[terra:SpatVector-class]{terra::SpatVector}}
object; \code{write_VECT} moves a \code{\link[terra:SpatVector-class]{terra::SpatVector}} object
through a temporary GeoPackage file to a GRASS vector object file.
\code{vect2neigh} returns neighbour pairs with shared boundary length as
described by Markus Neteler, in
\url{https://stat.ethz.ch/pipermail/r-sig-geo/2005-October/000616.html}.
\code{cygwin_clean_temp} can be called to try to clean the GRASS
mapset-specific temporary directory under cygwin.
}
\note{
Be aware that the GDAL-GRASS driver may have some
\href{https://github.com/OSGeo/gdal-grass/issues}{issues} for vector data. In
our experience, the error and warning messages for vector data can be
ignored. Further, the returned metadata about the coordinate reference system
may currently be incomplete, e.g. it may miss the EPSG code.
}
\examples{
# Run example if in active GRASS nc_basic_spm_grass7 location
run <- FALSE
if (nchar(Sys.getenv("GISRC")) > 0 &&
    read.dcf(Sys.getenv("GISRC"))[1, "LOCATION_NAME"] == "nc_basic_spm_grass7") {
  run <- require(terra, quietly = TRUE)
}

# Store original environment variable settings
GV <- Sys.getenv("GRASS_VERBOSE")
Sys.setenv("GRASS_VERBOSE" = 0)
ois <- get.ignore.stderrOption()
set.ignore.stderrOption(TRUE)

if (run) {
  # Create a new mapset
  meta <- gmeta()
  location_path <- file.path(meta$GISDBASE, meta$LOCATION_NAME)
  previous_mapset <- meta$MAPSET
  example_mapset <- "RGRASS_EXAMPLES"
  execGRASS("g.mapset", "c", mapset = example_mapset)
 }

if (run) {
  # Report basic metadata about the schools dataset
  execGRASS("v.info", map = "schools", layer = "1")
  print(vInfo("schools"))
 }

if (run) {
  # Read/write as a SpatVector
  schs <- read_VECT("schools")
  print(summary(schs))
 }

if (run) {
  try({
    write_VECT(schs, "newsch", flags = c("o", "overwrite"))
  })
  schs <- read_VECT("schools", use_gdal_grass_driver = FALSE)
 }

if (run) {
  write_VECT(schs, "newsch", flags = c("o", "overwrite"))
  execGRASS("v.info", map = "newsch", layer = "1")
 }

if (run) {
  nschs <- read_VECT("newsch")
  print(summary(nschs))
 }

if (run) {
  print(all.equal(names(nschs), as.character(vColumns("newsch")[, 2])))
 }

if (run) {
  # Show metadata for the roadsmajor dataset and read as spatVector
  print(vInfo("roadsmajor"))
 }

if (run) {
  roads <- read_VECT("roadsmajor")
  print(summary(roads))
}

# not run: vect2neigh() currently writes 3 new data sources in the PERMANENT
# mapset, despite this mapset not being the active one.
if (FALSE) {
  cen_neig <- vect2neigh("census")
  str(cen_neig)
}

# Cleanup the previously created datasets
if (run) {
  execGRASS(
    "g.remove",
    flags = "f",
    name = c("newsch", "newsch1"),
    type = "vector"
  )
  execGRASS("g.mapset", mapset = previous_mapset)
  if (example_mapset != previous_mapset) {
    unlink(file.path(location_path, example_mapset), recursive = TRUE)
  }
}

# Restore environment variable settings
Sys.setenv("GRASS_VERBOSE" = GV)
set.ignore.stderrOption(ois)
}
\author{
Roger S. Bivand, e-mail: \href{mailto:Roger.Bivand@nhh.no}{Roger.Bivand@nhh.no}
}
