% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter-fir-design.R
\name{design_filter_fir}
\alias{design_filter_fir}
\title{Design \code{'FIR'} filter using \code{\link{firls}}}
\usage{
design_filter_fir(
  sample_rate,
  filter_order = NA,
  data_size = NA,
  high_pass_freq = NA,
  high_pass_trans_freq = NA,
  low_pass_freq = NA,
  low_pass_trans_freq = NA,
  stopband_attenuation = 40,
  scale = TRUE,
  method = c("kaiser", "firls", "remez")
)
}
\arguments{
\item{sample_rate}{sampling frequency}

\item{filter_order}{filter order, leave \code{NA} (default) if undecided}

\item{data_size}{minimum length of data to apply the filter, used to
decide the maximum filter order. For 'FIR' filter, data length must be
greater than \code{3xfilter_order}}

\item{high_pass_freq}{high-pass frequency; default is \code{NA} (no
high-pass filter will be applied)}

\item{high_pass_trans_freq}{high-pass frequency band-width; default
is automatically inferred from data size.
Frequency \code{high_pass_freq - high_pass_trans_freq} is the corner
of the stop-band}

\item{low_pass_freq}{low-pass frequency; default is \code{NA} (no
low-pass filter will be applied)}

\item{low_pass_trans_freq}{low-pass frequency band-width; default
is automatically inferred from data size.
Frequency \code{low_pass_freq + low_pass_trans_freq} is the corner
of the stop-band}

\item{stopband_attenuation}{allowable power attenuation (in decibel) at
transition frequency; default is \code{40} dB.}

\item{scale}{whether to scale the filter for unity gain}

\item{method}{method to generate 'FIR' filter, default is using
\code{\link[gsignal]{kaiser}} estimate, other choices are
\code{\link{firls}} (with \code{hamming} window) and
\code{\link[gsignal]{remez}} design.}
}
\value{
'FIR' filter in 'Arma' form.
}
\description{
Design \code{'FIR'} filter using \code{\link{firls}}
}
\details{
Filter type is determined from \code{high_pass_freq} and
\code{low_pass_freq}. High-pass frequency is ignored if \code{high_pass_freq}
is \code{NA}, hence the filter is low-pass filter. When
\code{low_pass_freq} is \code{NA}, then
the filter is high-pass filter. When both \code{high_pass_freq} and
\code{low_pass_freq} are valid (positive, less than 'Nyquist'), then
the filter is a band-pass filter if band-pass is less than low-pass
frequency, otherwise the filter is band-stop.

Although the peak amplitudes are set at 1 by \code{low_pass_freq} and
\code{high_pass_freq}, the transition from peak amplitude to zero require
a transition, which is tricky but also important to set.
When 'FIR' filters have too steep transition boundaries, the filter tends to
have ripples in peak amplitude, introducing artifacts to the final signals.
When the filter is too flat, components from unwanted frequencies may also
get aliased into the filtered signals. Ideally, the transition bandwidth
cannot be too steep nor too flat. In this function, users may control
the transition frequency bandwidths via \code{low_pass_trans_freq} and
\code{high_pass_trans_freq}. The power at the end of transition is defined
by \code{stopband_attenuation}, with default value of \code{40} (i.e.
-40 dB, this number is automatically negated during the calculation).
By design, a low-pass 5 Hz filter with 1 Hz transition bandwidth results in
around -40 dB power at 6 Hz.
}
\examples{

# ---- Basic -----------------------------

sample_rate <- 500
data_size <- 1000

# low-pass at 5 Hz, with auto transition bandwidth
# from kaiser's method, with default stopband attenuation = 40 dB
filter <- design_filter_fir(
  low_pass_freq = 5,
  sample_rate = sample_rate,
  data_size = data_size
)

# Passband ripple is around 0.08 dB
# stopband attenuation is around 40 dB
print(filter)

diagnose_filter(
  filter$b, filter$a,
  fs = sample_rate,
  n = data_size,
  cutoffs = c(-3, -6, -40),
  vlines = 5
)

# ---- Advanced ---------------------------------------------

sample_rate <- 500
data_size <- 1000

# Rejecting 3-8 Hz, with transition bandwidth 0.5 Hz at both ends
# Using least-square (firls) to generate FIR filter
# Suggesting the filter order n=160
filter <- design_filter_fir(
  low_pass_freq = 3, low_pass_trans_freq = 0.5,
  high_pass_freq = 8, high_pass_trans_freq = 0.5,
  filter_order = 160,
  sample_rate = sample_rate,
  data_size = data_size,
  method = "firls"
)

#
print(filter)

diagnose_filter(
  filter$b, filter$a,
  fs = sample_rate,
  n = data_size,
  cutoffs = c(-1, -40),
  vlines = c(3, 8)
)



}
