% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psymet_functions.R
\name{gtheory}
\alias{gtheory}
\title{Generalizability Theory Reliability of a Score}
\usage{
gtheory(
  data,
  vrb.nm,
  ci.type = "classic",
  level = 0.95,
  cross.vrb = TRUE,
  R = 200L,
  boot.ci.type = "perc"
)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm}{character vector of colnames from \code{data} specifying the
variables/items.}

\item{ci.type}{character vector of length = 1 specifying the type of
confidence interval to compute. There are currently two options: 1)
"classic" = traditional ICC-based confidence intervals (see details), 2)
"boot" = bootstrapped confidence intervals.}

\item{level}{double vector of length 1 specifying the confidence level from 0
to 1.}

\item{cross.vrb}{logical vector of length 1 specifying whether the
variables/items should be crossed when computing the generalizability
theory coefficient. If TRUE, then only the covariance structure of the
variables/items will be incorperated into the estimate of reliability. If
FALSE, then the mean structure of the variables/items will be incorperated.}

\item{R}{integer vector of length 1 specifying the number of bootstrapped
resamples to use. Only used if \code{ci.type} = "boot".}

\item{boot.ci.type}{character vector of length 1 specifying the type of
bootstrapped confidence interval to compute. The options are 1) "perc" for
the regular percentile method, 2) "bca" for bias-corrected and accelerated
percentile method, 3) "norm" for the normal method that uses the
bootstrapped standard error to construct symmetrical confidence intervals
with the classic formula around the bias-corrected estimate, and 4) "basic"
for the basic method. Note, "stud" for the studentized method is NOT an
option. See \code{\link[boot]{boot.ci}} as well as
\code{\link{confint2.boot}} for details.}
}
\value{
double vector containing the generalizability theory coefficient,
  it's standard error (if \code{ci.type} = "boot"), and it's confidence
  interval.
}
\description{
\code{gtheory} uses generalizability theory to compute the reliability
coefficient of a score. It assumes single-level data where the rows are cases
and the columns are variables/items. Generaliability theory coefficients in
this case are the same as intraclass correlations (ICC). The default computes
ICC(3,k), which is identical to cronbach's alpha, from \code{cross.vrb} =
TRUE. When \code{cross.vrb} is FALSE, ICC(2,k) is computed, which takes mean
differences between variables/items into account. \code{gtheory} is a wrapper
function for \code{\link[psych]{ICC}}.
}
\details{
When \code{ci.type} = "classic" the confidence intervals are computed
according to the formulas laid out by McGraw, Kenneth, and Wong, (1996).
These are taken from the \code{\link[psych]{ICC}} function in the
\code{psych} package. They are appropriately non-symmetrical given ICCs are
not unbounded and range from 0 to 1. Therefore, there is no standard error
associated with the coefficient. Note, they differ from the confidence
intervals available in the \code{\link{cronbach}} function. When
\code{ci.type} = "boot" the standard deviation of the empirical sampling
distribution is returned as the standard error, which may or may not be
trustworthy depending on the value of the ICC and sample size.
}
\examples{

gtheory(attitude, vrb.nm = names(attitude), ci.type = "classic")
\dontrun{
gtheory(attitude, vrb.nm = names(attitude), ci.type = "boot")
gtheory(attitude, vrb.nm = names(attitude), ci.type = "boot",
   R = 250L, boot.ci.type = "bca")
}

# comparison to cronbach's alpha:
gtheory(attitude, names(attitude))
gtheory(attitude, names(attitude), cross.vrb = FALSE)
a <- suppressMessages(psych::alpha(attitude)[["total"]]["raw_alpha"])
psych::alpha.ci(a, n.obs = 30, n.var = 7, digits = 7) # slightly different confidence interval

}
\references{
McGraw, Kenneth O. and Wong, S. P. (1996), Forming inferences about some
intraclass correlation coefficients. Psychological Methods, 1, 30-46. + errata on page 390.
}
\seealso{
\code{\link{gtheorys}}
   \code{\link{gtheory_ml}}
   \code{\link{cronbach}}
}
