% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class.R
\name{quarto_object}
\alias{quarto_object}
\alias{quarto_div}
\alias{quarto_span}
\alias{quarto_tabset}
\alias{quarto_section}
\alias{quarto_group}
\alias{quarto_markdown}
\title{Dynamically generate quarto syntax}
\usage{
quarto_section(title, level)

quarto_tabset(content, level, title = NULL, names = NULL)

quarto_div(content, class = NULL, sep = "")

quarto_span(content, class = NULL, sep = "")

quarto_group(content, sep = "")

quarto_markdown(content, sep = "")
}
\arguments{
\item{title}{Character string specifying the text to use as a section title.
For \code{quarto_section()} this is a required argument. For \code{quarto_tabset()}
it is permitted to use \code{title = NULL}, in which case the tabset will
be printed without a section header above it. This is the default
behavior for tabsets.}

\item{level}{Numeric header level applied to section title or tabset names.
The \code{level} argument must be a whole number between 1 and 6. Only
relevant to quarto objects that produce section headings, specifically
\code{quarto_section()} and \code{quarto_tabset()}.}

\item{content}{List or character vector containing content to be included within
the quarto object. The expected format of the \code{content} argument
differs slightly depending on which function is used. See the
"details" section for more information.}

\item{names}{Character vector of names to be applied to the tabs in a tabset. Only
relevant to  \code{quarto_tabset()}. If \code{names = NULL}, the names will be
taken from the names of the \code{content} argument.}

\item{class}{Character vector specifying CSS classes to be applied to the content.
Only relevant to \code{quarto_div()} and \code{quarto_span()}. Defaults to
\code{class = NULL}, in which case the formatted text written to the document
will have a dummy CSS class "quartose-null" applied.}

\item{sep}{Character string specifying the separator to be used when merging content
for printing to the document. Defaults to \code{sep = ""} for all functions.}
}
\value{
These functions always return an object with parent S3 class
"quarto_object", in addition to a specific S3 class corresponding
to the function. For example, \code{quarto_section()} objects also possess the
"quarto_section" class.
}
\description{
Define quarto objects for insertion into a document.
Intended to be used inside a quarto document, within a knitr
code chunk with the \code{results: asis} option set.
}
\details{
The purpose of these functions is to allow the user to dynamically
generate quarto syntax from R. When used within a quarto document they
allow the user to generate callouts, margin text, tabsets, section
headers, and other kinds of quarto output. At the current state of
development the functionality is somewhat limited, discussed below.

The \verb{quarto_*()} functions supplied by the quartose package have a common
design: argument values supplied by the user are stored internally as a
list, with only a minimum of processing done at the time that the function
is called. The object is assigned to two S3 classes, the "quarto_object"
shared by all objects, and a specific class associated with the calling
function. These objects can be inspected and manipulated programmatically
like any other R objects prior to printing.

When creating a quarto object, note that most \verb{quarto_*()} functions take
a \code{content} argument, which differs slightly depending on the context:
\itemize{
\item For \code{quarto_section()} there is no `content`` argument: section headers
have titles, but they do not contain content.
\item For \code{quarto_span()} the `content`` argument \emph{must} be a character vector,
not a list.
\item For \code{quarto_div()} the \verb{content`` argument is permitted to be a character vector or a list, but it will always be stored internally as a list.  If the input is a list, it can contain other quarto objects. The  intended use for this is a div that contains several spans, but it is not limited to this use case. At present, }quarto_div()` cannot handle
plot objects, but functionality may be extended to permit this in future.
\item For \code{quarto_tabset()} the \code{content} argument \emph{must} be a list. The list
elements can be any printable R object: each element of the list will
appear in its own tab. At present the support for graphics objects is
limited: ggplot2 objects are captured and will only be rendered when
\code{knitr::knit_print()} is called. No attempt is made (as yet!) to support
other kinds of graphic objects, and if these are passed via the \code{content}
argument the function will likely fail.
\item For \code{quarto_markdown()} the \code{content} argument may be a character vector
or a list of character vectors. The function will throw an error if other
kinds of objects are passed via \code{content}.
\item For \code{quarto_group()} the \code{content} argument \emph{must} be a list, and all
elements of the list must be quarto objects. The intended use of this
function is simply to collect several quarto objects into a single group
that will be printed all at the same time rather than sequentially.
}

Creating a quarto object only defines the data structure, it does not
perform any formatting. Similarly, if the object is printed using
\code{print()}, no formatting will be applied. A brief summary of the
data structure will be printed to the console, no more. However, when
\code{knitr::knit_print()} is called, the quarto object is first passed to
the relevant \code{format()} method, which is responsible for constructing
the appropriate quarto syntax. Calling \code{format()} will return a
character vector or a list. If it returns a list all elements will
either be character strings with the appropriate quarto syntax, or a
plot object that has not yet been rendered. After formatting is applied
the \code{knitr::knit_print()} method will pass the strings (or plots) to
the document. For more detail on the formatting and printing methods
see \code{knit_print.quarto_object()} and \code{format.quarto_object()}.
}
\examples{
# quarto_section ------------------------------------------------------

sec <- quarto_section("A level-two header", level = 2L)

# quarto objects have two classes, a general purpose class shared by 
# all quarto objects, and a class specific to the function
class(sec) 
 
# base::print() displays an abstract summary of the object 
print(sec)

# knitr::knit_print() produces the rendered quarto syntax
knitr::knit_print(sec)

# quarto_span ---------------------------------------------------------

spn1 <- quarto_span("This is plain text")
spn2 <- quarto_span("This is underlined text", class = "underline")

print(spn1)

print(spn2)

knitr::knit_print(spn1)

knitr::knit_print(spn2)

# quarto_div ----------------------------------------------------------

# quarto_div objects are flexible: they can take a character vector as
# the content argument, but can also take lists of other objects; note
# that internally the content is always represented as a list
div1 <- quarto_div("This is a callout note", class = "callout-note")
div2 <- quarto_div(
  content = list(
    quarto_span(content = "You can wrap multiple spans in a div so that"),
    quarto_span(content = "some text is highlighted", class = "mark"),
    quarto_span(content = "and some is underlined", class = "underline")
  ),
  class = c("column-margin", "callout-tip"),
  sep = " "
)

print(div1)

print(div2)

knitr::knit_print(div1)

knitr::knit_print(div2)

# quarto_tabset -------------------------------------------------------

tbs <- quarto_tabset(list(tab1 = 1:10, tab2 = "hello"), level = 3L)

print(tbs)

knitr::knit_print(tbs)

# quarto_markdown -----------------------------------------------------

mkd <- quarto_markdown(list("- a markdown", "- list"), sep = "\n")

print(mkd)

knitr::knit_print(mkd)

# quarto_group --------------------------------------------------------

grp <- quarto_group(list(
  quarto_div("This is a callout note", class = "callout-note"),
  quarto_div("This is a callout tip", class = "callout-tip")
))

print(grp)

knitr::knit_print(grp)
}
