## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----basic_example------------------------------------------------------------
library(pridit)

# Create sample healthcare quality data
healthcare_data <- data.frame(
  Hospital_ID = c("A", "B", "C", "D", "E"),
  Smoking_cessation = c(0.9, 0.85, 0.89, 1.0, 0.89),
  ACE_Inhibitor = c(0.99, 0.92, 0.90, 1.0, 0.93),
  Proper_Antibiotic = c(1.0, 0.99, 0.98, 1.0, 0.99)
)

print(healthcare_data)

## ----ridit_step---------------------------------------------------------------
# Calculate ridit scores
ridit_scores <- ridit(healthcare_data)
print(ridit_scores)

## ----weights_step-------------------------------------------------------------
# Calculate PRIDIT weights
weights <- PRIDITweight(ridit_scores)
print(weights)

## ----final_scores-------------------------------------------------------------
# Calculate final PRIDIT scores
final_scores <- PRIDITscore(ridit_scores, healthcare_data$Hospital_ID, weights)
print(final_scores)

## ----test_dataset-------------------------------------------------------------
# Load the test dataset
data(test)
print(test)

# Run the complete analysis
ridit_result <- ridit(test)
weights <- PRIDITweight(ridit_result)
final_scores <- PRIDITscore(ridit_result, test$ID, weights)

print(final_scores)

## ----quality_example----------------------------------------------------------
# Hospital quality assessment example
hospital_quality <- data.frame(
  Hospital = paste0("Hospital_", 1:10),
  Mortality_Rate = c(0.02, 0.03, 0.01, 0.04, 0.02, 0.03, 0.01, 0.02, 0.05, 0.01),
  Readmission_Rate = c(0.10, 0.12, 0.08, 0.15, 0.09, 0.11, 0.07, 0.10, 0.16, 0.08),
  Patient_Satisfaction = c(8.5, 7.2, 9.1, 6.8, 8.0, 7.5, 9.3, 8.2, 6.5, 9.0),
  Safety_Score = c(85, 78, 92, 70, 82, 79, 94, 86, 68, 90)
)

# Note: For this example, we'll need to invert mortality and readmission rates
# since lower values indicate better quality
hospital_quality$Mortality_Rate <- 1 - hospital_quality$Mortality_Rate
hospital_quality$Readmission_Rate <- 1 - hospital_quality$Readmission_Rate

# Calculate PRIDIT scores
ridit_scores <- ridit(hospital_quality)
weights <- PRIDITweight(ridit_scores)
quality_scores <- PRIDITscore(ridit_scores, hospital_quality$Hospital, weights)

# Sort by PRIDIT score
quality_ranking <- quality_scores[order(quality_scores$PRIDITscore, decreasing = TRUE), ]
print(quality_ranking)

## ----variable_importance------------------------------------------------------
# Create a data frame showing variable importance
variable_names <- colnames(hospital_quality)[-1]  # Exclude ID column
importance_df <- data.frame(
  Variable = variable_names,
  Weight = weights,
  Abs_Weight = abs(weights)
)

# Sort by absolute weight to see most important variables
importance_df <- importance_df[order(importance_df$Abs_Weight, decreasing = TRUE), ]
print(importance_df)

## ----longitudinal_example-----------------------------------------------------
# Simulate hospital performance over two time periods
hospitals <- paste0("Hospital_", 1:5)

# Time 1 data
time1_data <- data.frame(
  Hospital = hospitals,
  Quality_A = c(0.85, 0.90, 0.78, 0.92, 0.88),
  Quality_B = c(0.82, 0.85, 0.80, 0.88, 0.84),
  Quality_C = c(0.90, 0.87, 0.85, 0.91, 0.86)
)

# Time 2 data
time2_data <- data.frame(
  Hospital = hospitals,
  Quality_A = c(0.88, 0.91, 0.82, 0.93, 0.85),
  Quality_B = c(0.85, 0.87, 0.83, 0.89, 0.82),
  Quality_C = c(0.92, 0.88, 0.87, 0.93, 0.88)
)

# Calculate PRIDIT scores for both time periods
time1_ridit <- ridit(time1_data)
time1_weights <- PRIDITweight(time1_ridit)
time1_scores <- PRIDITscore(time1_ridit, time1_data$Hospital, time1_weights)

time2_ridit <- ridit(time2_data)
time2_weights <- PRIDITweight(time2_ridit)
time2_scores <- PRIDITscore(time2_ridit, time2_data$Hospital, time2_weights)

# Combine results for comparison
longitudinal_results <- merge(time1_scores, time2_scores, by = "Claim.ID", suffixes = c("_Time1", "_Time2"))
longitudinal_results$Change <- longitudinal_results$PRIDITscore_Time2 - longitudinal_results$PRIDITscore_Time1

print(longitudinal_results)

