% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/example.R
\name{Example}
\alias{Example}
\title{Find combination of parameters required for achieving a desired power
(or other objective).}
\usage{
Example(
  x,
  example = NULL,
  target_value = NULL,
  target_at_least = TRUE,
  find_lowest = TRUE,
  method = "step",
  summary_function = mean
)
}
\arguments{
\item{x}{Object of class \code{power_array}}

\item{example}{List with named elements representing the constellation of
parameter values for which the example should be found. The names of this
list should match the dimension names of \code{x}, their values should be exact
values available at these dimensions. See example for an illustration.}

\item{target_value}{Which value (of typically power) should be achieved at
the example.}

\item{target_at_least}{Logical. Set to TRUE if you aim to achieve a minimum
value (e.g., a power must be \emph{at least} 90\%), or FALSE if you want to
allow a maximum value (e.g., the width of the expected CI may be \emph{at most}
a certain value).}

\item{find_lowest}{Logical, indicating whether the example should be found
that minimizes a parameter (typically: minimal required n) to achieve the
\code{target_value} or maximizes this assumption (e.g., maximal allowed SD).}

\item{method}{Character string, indicating how the location of the example
is found, passed on internally to \code{FindTarget}. Either "step": walking in
steps along the parameter of interest or "lm": Interpolating assuming a
linear relation between the parameter of interest and (qnorm(x) + qnorm(1
\itemize{
\item 0.05)) ^ 2. This method "lm" is inspired on the implementation in the
sse package by Thomas Fabbro.
}}

\item{summary_function}{When x' attribute \code{summarized} is FALSE, x is
summarized across iterations using this function before searching the
example.}
}
\value{
Example returns a list containing:
\itemize{
\item "requested_example": the parameter combination at which the power (or
whatever the values represent) was searched to achieve level \code{target_value}
(typically the minimal power, e.g., .9), searching along parameter \verb{required name} (typically n).
\item "objective": was \code{required_name} searched to find the "min" or "max" of
x?
\item "target_value": which value should the power (or any other value) have?
\item "required_name": the parameter searched along to find the minimum (or
maximized if slot \code{searched} = 'max') to achieve objective. (typically n)
\item "required_value": the minimum (or maximum if \code{searched} = "max") for
parameter \code{required_name} (which is typically n)
\item "searched": was the "min" or "max" for \code{required_name} searched?
\item "target_at_least": Is the target_value a minimum (TRUE, as typical for power) or a
maximum (FALSE, e.g., an expected uncertainty level)?
}
}
\description{
Find combination of parameters yielding desired power (or any other target
value) in an object of class "power_array".
}
\details{
In the most typical use case, and this is also the default, \code{Example}
searches the \emph{minimal} n where the power is \emph{at least} equal to the value
given by argument \code{target}. The function is, however, designed much more
generically. The explanation below may be less helpful than trying the
examples, but for completeness:

Argument \code{example} slices out a vector from object \code{x}, representing the
values at the parameter combination given in example, thus, along the
remaining parameter. Then, \code{Example} searches along this vector for the
\emph{minimal} parameter value where the value of the vector is \emph{at least} equal
to \code{target}. Thus, if the sliced out vector contains values of "power" along
the parameter "effect size", it searches the minimal effect size at which
the target power is achieved.

Two complications are made to allow for complete flexibility:
\enumerate{
\item In the above description, \emph{minimal} can be changed to \emph{maximal} by
setting argument \code{find_lowest} to FALSE. This is useful in the situation where
one, e.g., searches for the highest standard deviation at which it is still
possible to find a desirable power.
\item In the above description, \emph{at least} can be changed to \emph{at most} by
setting \code{target_at_least} to FALSE. This allows to search, for example, for
the minimal sample size where the expected confidence interval is smaller
than a certain desired width.
}

Example searches for the minimum or maximum on one parameters (say, the
minimum n) given \emph{one single constellation} of further parameters. However,
you may want to study how, say, the required n (or any other value) depends
on the value of further parameters. The functions PowerPlot and GridPlot
offer plotting functionalities to graphically illustrate such
dependencies. If you want to find "Examples" as a function of parameter
settings and work with these, you can use the workhorse behind 'Example',
PowerPlot and Gridplot, \code{\link{FindTarget}}
}
\examples{
## ============================================
## Typical use case: find lowest n for a certain target power
## ============================================
sse_pars = list(
  n = seq(from = 10, to = 60, by = 2),
  delta = seq(from = 0.5, to = 1.5, by = 0.1), ## effect size
  sd = seq(.1, .9, .2)) ## Standard deviation
PowFun <- function(n, delta, sd){
  ptt = power.t.test(n = n/2, delta = delta, sd = sd,
                     sig.level = 0.05)
  return(ptt$power)
}
power_array = PowerGrid(pars = sse_pars, fun = PowFun, n_iter = NA)
##'
ex_out = Example(power_array,
                 example = list(delta = .7, sd = .7),
                 target_value = .9)
ex_out #

## ============================================
## Illustration argument `find_lowest`
## ============================================
##
## In this example, we search for the *highest sd* for which the power is at
## least .9.
ex_out = Example(power_array,
                 example = list(n = 40, delta = .7),
                 target_value = .9, find_lowest = FALSE)
ex_out # note how the printed result indicates it searched for a maximal
                                        # permissible sd.

## ============================================
## Illustration argument `target_at_least`
## ============================================
##
## In the example below, we search for the lowest n where the expected CI-width
## is not larger than .88.
PowFun <- function(n, delta, sd){
  x1 = rnorm(n = n/2, sd = sd)
  x2 = rnorm(n = n/2, mean = delta, sd = sd)
  CI_width = diff(t.test(x1, x2)$conf.int) # CI95 is saved
}
sse_pars = list(
  n = seq(from = 10, to = 60, by = 5),
  delta = seq(from = 0.5, to = 1.5, by = 0.2),
  sd = seq(.5, 1.5, .2))
## we iterate, and take the average across iterations to get expected CI-width:
n_iter = 20
set.seed(1)
power_array = PowerGrid(pars = sse_pars, fun = PowFun, n_iter = n_iter)
summary(power_array)
## Now, find lowest n for which the average CI width is *smaller than .88*.
ex_out = Example(power_array,
                 example = list(delta = .7, sd = .7),
                 target_value = .88,
                 find_lowest = TRUE, # we search the *lowest* n
                 target_at_least = FALSE # for a *maximal* mean CI width
                 )
ex_out # note how the printed result indicates the target CI is a maximum.

## ============================================
## When both `find_lowest` and `target_at_least` are FALSE
## ============================================
##
## In this example, we search for the *highest sd* for which the average CI
## width is still *smaller than or equal to .88*.
ex_out = Example(power_array,
                 example = list(delta = .7, n = 60),
                 target_value = .88,
                 find_lowest = FALSE, # we search the *highest* sd
                 target_at_least = FALSE # for a *maximal* mean CI width
                 )

ex_out # note how the printed result indicates that the *maximal permissible SD*
       # was found for a CI of *at most .88*.

}
\seealso{
\code{\link{PowerGrid}}, \code{\link{FindTarget}},
\code{\link{PowerPlot}}, \code{\link{GridPlot}}
}
\author{
Gilles Dutilh
}
