# change name of a and g to alpha and gamma
# alpha und gamma k?nnen f?r alle Arten unterschiedlich sein
# statt spacing k?nnte man auch ein Argument fun einf?hren, f?r regular eine
# Funktion schreiben und sonst runif oder rnorm


"make.species" <- function(nspp = 30, Amax, srange,
                           fun, xpar, alpha = 4, gamma = 4) {
  # maximum Ao
  if (is.numeric(Amax)) {
    Ao <- Amax
  } else {
    Ao <- Amax(n = nspp)
  }
  r <- srange
  # else gamma=gamma
  # mean position m
  if (missing(fun)) {
    m <- seq(xpar[1], xpar[2], length.out = nspp)
  } else {
    m <- fun(nspp, xpar[1], xpar[2])
  }
  spp <- data.frame(Ao = Ao, m = m, r = r, alpha = alpha, gamma = gamma)
  list(spp = spp)
}



#' @title Generates species response parameters for n dimensions
#' @description
#' Generates species response parameters to n environmental variables following
#' Minchin (1987).
#' @param nspp Number of species to be generated.
#' @param Amax Maximum abundance of a species.
#' `Amax` currently allows three options:
#' i) a function how to generate maximum abundances (e.g. `runif`, `rgamma`)
#' ii) a vector of length `nspp`
#' iii) a single number that is used as maximum abundance for all the species.
#' @param fun Function to generate species optima (e.g. `rnorm`, `runif`).
#' The two parameters in `xpar` are passed to function `fun`.
#' If omitted species optima are generated at regular intervals between the two
#' values in `xpar`.
#' @param xpar Two numbers describing a distribution
#' e.g mu and sigma for a normal distribution,
#' lower and upper bound for a random uniform distribution.
#' @param srange Length of the ecological gradient to which individual species
#' respond. Either one number or a matrix with `nspp` rows and `ndim` columns.
#' If `srange` should be different for different environmental variables a
#' simpler solution is to change argument `elen` in \code{\link{make.env}}
#' accordingly. E.g. `elen = c(100,50,50)` when using three environmental
#' gradients.
#' @param alpha Shape parameter of the beta distribution.
#' One number or a matrix with `nspp` rows and `ndim` columns.
#' @param gamma Shape parameter of the beta distribution.
#' One number or a matrix with `nspp` rows and `ndim` columns.
#' @param ndim Number of environmental variables to which generated species
#' should respond.
#' @param sdistr Users may supply own distributions of species optima.
#' Matrix with `nspp` rows and `ndim` columns
#' (as well in the special case of `ndim = 1`).
#' @param ocor Correlation matrix of the species optima.
#' May be generated by code {\link{cor.mat.fun}}.
#' @param odistr Distribution of the correlated optima either 'uniform' or
#' 'Gaussian'
#'
#' @details
#' Details on the exact generation of species response functions from parameters
#' `Amax`, `m`, `r`, `gamma` and `alpha` are given in Minchin (1987).
#' Species response curves are determined by five parameters:
#' a parameter determining the maximum abundance (`Amax`)
#' and one describing the location (`m`) of this mode.
#' A parameter determining to which environmental range the species respond
#' (`srange` in the input `r` in the output) and two parameters
#' (`alpha`, `gamma`) describing the shape of the species response function.
#' If `alpha` = `gamma` the response curve is symmetric
#' (`alpha` = `gamma` = 4, yields approximately Gaussian distributions).
#' Additionally, species optima for several environmental variables may be
#' correlated.
#' Currently this is only possible for Gaussian or uniform distributions of
#' species optima. Users may as well supply previously generated optima
#' (e.g. optima similar to a real dataset).
#'
#' @returns
#' List with `ndim` elements. Each list contains the species response
#' parameters to one environmental gradient.
#' @references  Minchin, P.R. (1987) Multidimensional Community Patterns:
#' Towards a Comprehensive Model. _Vegetatio_, **71**, 145-156.
#' \doi{10.1007/BF00039167}
#' @author Mathias Trachsel and Richard J. Telford
#' @examples
#' spec.par <- species(
#'   nspp = 30, Amax = runif, srange = 200, fun = runif,
#'   xpar = c(-50, 150),
#'   ndim = 5, alpha = 4, gamma = 4
#' )
#' spec.par <- species(
#'   nspp = 30, ndim = 3, Amax = runif, xpar = c(-50, 150),
#'   srange = 200, alpha = 4, gamma = 4
#' )
#'
#' # example where srange, alpha and gamma are different for each
#' # species and environmental gradient.
#' spec.par <- species(
#'   nspp = 30, ndim = 3, Amax = runif, xpar = c(-50, 150),
#'   srange = matrix(ncol = 3, runif(90, 100, 200)),
#'   alpha = matrix(ncol = 3, runif(90, 1, 5)),
#'   gamma = matrix(ncol = 3, runif(90, 1, 5))
#' )
#'
#' # example where species optima are correlated
#' correlations <- list(c(1, 2, 0.5), c(1, 3, 0.3), c(2, 3, 0.1))
#' spec.cor.mat <- cor.mat.fun(3, correlations)
#' spec.par <- species(
#'   nspp = 30, ndim = 3, Amax = runif, xpar = c(50, 50),
#'   srange = 200, alpha = 4, gamma = 4,
#'   ocor = spec.cor.mat, odistr = "Gaussian"
#' )
#'
#' # example for species response curves (users should alter alpha and gamma)
#' spec.par <- species(
#'   nspp = 1, Amax = 200, srange = 200, fun = runif,
#'   xpar = c(50, 50),
#'   ndim = 1, alpha = 3, gamma = 1
#' )
#' env <- -50:150
#' response <- palaeoSig:::make.abundances(
#'   env = -50:150,
#'   param = spec.par[[1]]$spp
#' )
#' plot(env, response, type = "l")
#' @seealso \code{\link{make.env}}, \code{\link{abundances}}
#' @keywords datagen
#' @importFrom stats pnorm
#' @importFrom MASS mvrnorm
#' @export
#'
species <- function(nspp = 30, Amax, fun, xpar, srange, alpha = 4, gamma = 4,
                    ndim, sdistr, ocor, odistr) {
  if (length(xpar) == 2) {
    xpar <- matrix(ncol = 2, rep(xpar, ndim), byrow = TRUE)
  }
  if (length(alpha) == 1) {
    alpha <- matrix(ncol = ndim, rep(alpha, ndim * nspp))
  }
  if (length(gamma) == 1) {
    gamma <- matrix(ncol = ndim, rep(gamma, ndim * nspp))
  }
  if (length(srange) == 1) {
    srange <- matrix(ncol = ndim, rep(srange, ndim * nspp))
  }
  if (length(Amax) == 1 && is.numeric(Amax)) {
    srange <- matrix(ncol = 1, rep(Amax, nspp))
  }
  if (missing(fun)) {
    spp <- lapply(1:ndim, function(x) {
      spp <- make.species(
        nspp = nspp, Amax = Amax, srange = srange[, x],
        xpar = xpar[x, ], alpha = alpha[, x],
        gamma = gamma[, x]
      )
      spp <- spp$spp[sample(1:nspp), ]
    })
  } else {
    spp <- lapply(1:ndim, function(x) {
      spp <- "make.species"(nspp = nspp, Amax = Amax, fun = fun,
        srange = srange[, x], xpar = xpar[x, ],
        alpha = alpha[, x], gamma = gamma[, x])
    })
  }

  if (!missing(ocor)) {
    if (odistr == "Gaussian") {
      # producing gaussian correlated variable
      opt <- mvrnorm(
        n = nspp, mu = rep(0, ndim), Sigma = ocor,
        empirical = FALSE
      )
      #    adjusting mean and variance to desired values
      opt <- sapply(1:ndim, function(x) {
        opt[, x] * xpar[x, 2] + xpar[x, 1]
      })
    }

    if (odistr == "uniform") {
      # producing gaussian correlated variable
      opt <- mvrnorm(
        n = nspp, mu = rowMeans(xpar), Sigma = ocor,
        empirical = FALSE
      )
      # inverse quantile function produces uniform correlated data
      opt <- pnorm(scale(opt))
      aa <- t(t(opt) * apply(xpar, 1, diff))
      opt <- apply(aa, 2, function(y) {
        y + (colMeans(t(xpar)) - colMeans(aa))
      })
    }

    spp <- lapply(1:ndim, function(x) {
      spp[[x]]$m <- opt[, x]
      spp[[x]]
    })
  }
  if (!missing(sdistr)) {
    spp <- lapply(1:ndim, function(x) {
      spp[[x]]$m <- sdistr[, x]
      spp[[x]]
    })
  }
  spp
}

"run.spp.present" <- function(spp) {
  spp[, apply(spp, 2, sum) > 0]
}
#-------------------------------------------------------------------------------






#-------------------------------------------------------------------------------
#' @title Generate correlation matrix
#' @description
#' Generates a correlation matrix for the environmental variables generated in
#' \code{\link{make.env}} and for
#' correlated species optima in \code{\link{species}}.
#' Only used when correlated environmental variables or optima are generated.
#'
#' @param ndim Number of environmental variables that are subsequently generated
#' with \code{\link{make.env}}.
#' @param cors List of correlations between environmental variables.
#' Each element of the list consists of three numbers,
#' the first two numbers indicate the variables that are correlated,
#' the third number is the correlation coefficient.
#' If correlations between two variables are omitted the correlation remains 0.

#' @returns A correlation matrix
#' @author Mathias Trachsel
#' @seealso \code{\link{make.env}}, \code{\link{species}}
#' @examples
#' correlations <- list(c(1, 2, 0.5), c(1, 4, 0.1), c(2, 5, 0.6))
#' cor.mat <- cor.mat.fun(5, correlations)
#' @keywords datagen

#' @export
cor.mat.fun <- function(ndim, cors) {
  cor_mat <- diag(ndim)
  if (!missing(cors)) {
    for (i in seq_along(cors)) {
      cor_mat[cors[[i]][1], cors[[i]][2]] <- cors[[i]][3]
      cor_mat[cors[[i]][2], cors[[i]][1]] <- cors[[i]][3]
    }
  }
  cor_mat
}


#--------------------------------------------------------------------------
#' @title Generates simulated environmental variables
#' @description
#' Simulates environmental variables used for generating species abundances.
#' Environmental variables may be correlated,
#' and may follow different distributions.
#' @param n Number of samples to be generated.
#' @param elen Range of the environmental variables.
#' Single number or vector of length `ndim`.
#' @param emean Mean of the environmental variables.
#' Single number or vector of length `ndim`.
#' @param edistr Distribution of the environmental variables.
#' Currently 'uniform' and 'Gaussian' are supported.
#' @param ecor Correlation matrix of the environmental variables.
#' Object can be generated with \code{\link{cor.mat.fun}}.
#' If omitted environmental variables are not correlated.
#' @param ndim Number of environmental variables to generate.
#' @returns Matrix of environmental variables. `n` rows and `ndim` columns.
#' @references  Minchin, P.R. (1987) Multidimensional Community Patterns:
#' Towards a Comprehensive Model. _Vegetatio_, **71**, 145-156.
#' \doi{10.1007/BF00039167}
#' @author Mathias Trachsel and Richard J. Telford
#' @seealso \code{\link{cor.mat.fun}}
#' @examples
#' env.vars <- make.env(100,
#'   elen = rep(100, 10), emean = rep(50, 10),
#'   edistr = "uniform", ndim = 10
#' )
#' @keywords datagen
#' @export

make.env <- function(n, elen, emean, edistr, ecor, ndim) {
  if (missing(ecor)) {
    ecor <- diag(ndim)
  }
  if (edistr == "uniform") {
    env_norm <- mvrnorm(
      n = n, mu = rep(0, ndim), Sigma = ecor,
      empirical = FALSE
    )
    env <- pnorm(scale(env_norm))
    env <- sapply(1:ndim, function(x) {
      aa <- env[, x] * elen[x]
      aa + (emean[x] - mean(aa))
    })
  }
  if (edistr == "Gaussian") {
    env <- mvrnorm(n = n, mu = rep(0, ndim), Sigma = ecor, empirical = FALSE)
    env <- apply(env, 2, function(x) {
      (x - min(x)) / (max(x) - min(x))
    })
    env <- sapply(1:ndim, function(x) {
      aa <- env[, x] * elen[x]
      aa + (emean[x] - mean(aa))
    })
  }
  env <- as.data.frame(env)
  env
}


#-------------------------------------------------------------------------------
# make abundances

make.abundances <- function(env, param) {
  env <- as.matrix(env)
  # error checking
  stopifnot(ncol(env) == nrow(param))

  param <- as.data.frame(param)
  with(param, {
    # calculate b and d
    d <- numeric(ncol(env))
    b <- numeric(ncol(env))
    for (k in seq_len(ncol(env))) {
      b[k] <- alpha[k] / (alpha[k] + gamma[k])
      d[k] <- b[k]^alpha[k] * (1 - b[k])^gamma[k]
    }
    d <- prod(d)
    # find which env values are within taxon's range
    in_range <- matrix(numeric(0), nrow = nrow(env), ncol = ncol(env))
    for (k in seq_len(ncol(env))) {
      c1 <- m[k] - r[k] * b[k] <= env[, k]
      c2 <- m[k] + r[k] * (1 - b[k]) >= env[, k]
      in_range[, k] <- c1 & c2
    }
    in_range <- apply(in_range, 1, function(x) {
      all(as.logical(x))
    })

    # calculate abundances
    A <- numeric(nrow(env))
    A[!in_range] <- 0
    xx <- env[in_range, , drop = FALSE]
    if (nrow(xx) > 0) {
      h <- matrix(numeric(0), nrow(xx), ncol(xx))
      for (k in seq_len(ncol(xx))) {
        h[, k] <- ((xx[, k] - m[k]) / r[k] + b[k])^alpha[k] *
          (1 - ((xx[, k] - m[k]) / r[k] + b[k]))^gamma[k]
      }
      A[in_range] <- Ao[1] / d * apply(h, 1, prod)
    }
    A
  })
}
#-------------------------------------------------------------------------------
# testen von make abundances
"add.noise.bs" <- function(spp, cnt) {
  spp <- spp / rowSums(spp) # percent data
  mat <- apply(spp, 1, function(sam) {
    sam <- sample(seq_len(ncol(spp)), round(cnt), replace = TRUE, prob = sam)
    # ,sample(1:ncol(spp),round(cnt/2), replace=TRUE)) # add more noise poisson
    # count - include 1:n to ensure no taxa skipped by table as empty
    table(c(sam, seq_len(ncol(spp))))
  })
  t(mat - 1)
}


#-----------------------------------------------------------------
#' @title Generates simulated species abundances
#' @description Generates species abundances based on species response functions
#' and environmental variables.
#' @param env Environmental variables.
#' Usually generated by \code{\link{make.env}}.
#' Users may as well supply own environmental variables.
#' @param spp Species parameters. Usually generated by \code{\link{species}}.
#' @param  nc  Count sum to be simulated.
#' If omitted no simulation of the counting process is carried out.
#' @returns A list with two elements:
#' \item{spp}{A data frame with species abundances.}
#' \item{env}{A data frame with environmental variables.}
#' @references  Minchin, P.R. (1987) Multidimensional Community Patterns:
#' Towards a Comprehensive Model. _Vegetatio_, **71**, 145-156.
#' \doi{10.1007/BF00039167}
#' @author Mathias Trachsel and Richard J. Telford
#' @seealso \code{\link{make.env}}, \code{\link{species}}
#' @examples
#' spec <- species(
#'   nspp = 30, ndim = 10, Amax = runif, fun = runif,
#'   xpar = c(-50, 150), srange = 200, alpha = 4, gamma = 4
#' )
#' env.var <- make.env(100,
#'   elen = rep(100, 10), emean = rep(50, 10),
#'   edistr = "uniform", ndim = 10
#' )
#' spec.abun <- abundances(env.var, spec, 200)

#' @keywords datagen
#' @export
abundances <- function(env, spp, nc) {
  ndim <- ncol(env)
  spp <- matrix(ncol = 5 * ndim, unlist(spp))
  colnames(spp) <- rep(c("Ao", "m", "r", "alpha", "gamma"), ndim)
  abun <- sapply(seq_len(nrow(spp)), function(x) {
    param <- matrix(ncol = 5, spp[x, ], byrow = TRUE)
    colnames(param) <- c("Ao", "m", "r", "alpha", "gamma")
    make.abundances(env = env, param = param)
  })
  if (!missing(nc)) {
    abun <- add.noise.bs(abun, nc)
  }
  colnames(abun) <- seq_len(ncol(abun))
  abun <- run.spp.present(abun)
  abun <- abun / rowSums(abun)
  list(spp = as.data.frame(abun), env = env)
}


#------------------------------------------------------------------------------
#' @title Simulate species data sets
#' @description
#' Function to simulate species data following Minchin (1987).
#' This functions generates species response functions,
#' simulates environmental variables and simulates species assemblages
#' based on species response functions and environmental variables.
#' Users can supply own species parameters
#' (e.g. when simulating calibration and fossil datasets) and
#' own environmental variables.
#' @param ndim Number of environmental variables to generate.
#' @param n Number of samples to be generated.
#' @param elen Range of the environmental variables.
#' Single number or vector of length `ndim`.
#' @param emean Mean of the environmental variables.
#' Single number or vector of length `ndim`.
#' @param edistr Distribution of the environmental variables.
#' Currently 'uniform' and 'Gaussian' are supported.
#' @param ecor Correlation matrix of the environmental variables.
#' Object generated by \code{\link{cor.mat.fun}}.
#' If omitted environmental variables are uncorrelated.
#' @param cnt Count sum to be simulated.
#' @param spec Users may supply their own species parameters.
#' @param env  Users may supply their own environmental variables.
#' @param \dots Arguments passed to \code{\link{species}}

#' @returns
#' List with three elements:
#' \item{spp}{ Species abundance data.}
#' \item{env}{ Environmental variables used to simulate species abundance data.}
#' \item{spec}{ Species parameters.}

#' @references  Minchin, P.R. (1987) Multidimensional Community Patterns:
#' Towards a Comprehensive Model. _Vegetatio_, **71**, 145-156.
#' \doi{10.1007/BF00039167}
#' @author Mathias Trachsel and Richard J. Telford
#' @seealso \code{\link{make.env}}, \code{\link{species}},
#' \code{\link{cor.mat.fun}}
#' @examples
#' calib <- make.set(
#'   nspp = 90, ndim = 3, Amax = runif, fun = runif,
#'   xpar = c(-50, 150), srange = 400, alpha = 4, gamma = 4,
#'   n = 100, elen = rep(100, 3), emean = rep(50, 3),
#'   edistr = "uniform", cnt = 1000
#' )
#'
#' # Provide species parameters generated above, so that the fossil data use the
#' #    same species parameters.
#' fos <- make.set(
#'   ndim = 3, n = 100, elen = rep(100, 3), emean = rep(50, 3),
#'   edistr = "uniform", cnt = 1000, spec = calib$spec
#' )
#'
#' # Supplying own environmental variables and species parameters.
#' env.vars <- make.env(100,
#'   elen = rep(100, 3), emean = rep(50, 3),
#'   edistr = "uniform", ndim = 3
#' )
#' fos <- make.set(cnt = 1000, spec = calib$spec, env = env.vars)
#'
#' @keywords datagen
#' @export

make.set <- function(ndim, n, elen, emean, edistr, ecor, cnt, spec, env, ...) {
  if (missing(spec)) {
    spec <- species(ndim, ...)
  }
  if (missing(env)) {
    env <- make.env(n, elen, emean, edistr, ecor, ndim)
  }
  dat <- abundances(env, spec, cnt)
  list(
    spp = dat$spp,
    env = dat$env,
    spec = spec
  )
}
