#' Which transpositions give elementary voice leadings?
#'
#' Just as the transpositions of the diatonic scale can be generated by
#' Hook (2008)'s <doi:10.1515/9781580467476-008> elementary
#' "signature transformation," the transpositional voice leadings of any
#' set can generally be decomposed into a small number of basic motions.
#' These motions correspond to the arrows in a set's [brightnessgraph()].
#' (The qualifier "generally" is needed because of certain problematic edge
#' cases, such as the perfectly even scales of [edoo()] whose minimal voice
#' leadings always involve entirely parallel motion, which cannot be 
#' derived from "mode shift" voice leadings represented on a brightness graph.)
#' `vl_generators()` identifies these basic voice-leading motions.
#'
#' @inheritParams tnprime
#' @inheritParams minimize_vl
#'
#' @returns 2-by-m matrix whose m columns represent the m distinct voice-leading
#'   generators. The top row indicates the generic size of each interval; the 
#'   bottom row indicates the specific size. Results are sorted so that the first
#'   row (generic intervals) is strictly increasing.
#'
#' @examples
#' diatonic_scale <- c(0, 2, 4, 5, 7, 9, 11)
#' melodic_minor <- c(0, 2, 3, 5, 7, 9, 11)
#' vl_generators(diatonic_scale)
#' vl_generators(melodic_minor)
#' vl_generators(j(dia))
#'
#' maj7 <- c(0, 4, 7, 11)
#' vl_generators(maj7)
#'
#' @export
vl_generators <- function(set, edo=12, rounder=10) {
  tiny <- 10^(-1 * rounder)
  card <- length(set)

  if (evenness(set, edo=edo) < tiny) {
    warning("Perfectly even scales have no voice-leading generators")
    return(matrix(c(numeric(0), numeric(0)), nrow=2))
  }

  scalar_interval_matrix <- sim(set, edo=edo)

  reduced_comparisons <- bg_reduction(set=set, edo=edo, rounder=rounder)
  arrows_in_graph  <- which(reduced_comparisons==1, arr.ind=TRUE)
  from_which_mode <- arrows_in_graph[,1]
  generic_intervals <- (arrows_in_graph[,2] - from_which_mode) %% card
  generic_intervals <- (generic_intervals + 1)
  specific_intervals <- scalar_interval_matrix[cbind(generic_intervals, from_which_mode)]
  
  # Reformat the intervals so their display value is more human readable
  generic_intervals <- (-1*(generic_intervals-1)) %% card
  specific_intervals <- (-1 * specific_intervals) %% edo

  duplications <- duplicated(round(specific_intervals, digits=rounder))
  res <- rbind(generic_intervals, specific_intervals)
  res <- res[, !duplications]
  
  res <- insist_matrix(res)

  res <- res[,order(res[1,])]
  insist_matrix(res)
}

#' Elementary voice leadings
#'
#' Calculates the "**v**oice-**l**eading **sig**nature" of the set's elementary transpositions
#' as determined by [vl_generators()]. 
#'
#' Note that the voice leadings determined by `vlsig()` can be different from the
#' corresponding ones at the same \eqn{T_n} level in [vl_rolodex()]. The latter function
#' prioritizes minimal voice leadings, whereas `vlsig()` prioritizes *elementary* voice
#' leadings derived from a set's [brightnessgraph()]. In particular, this means that
#' `vlsig()` voice leadings will always be ascending, involve at least one common tone,
#' and involve no contrary motion. See the `odd_pentachord` voice leadings in the Examples.
#' 
#'
#' @inheritParams vl_generators
#' @inheritParams ifunc
#' @param index Integer: which voice-leading generator should be displayed? Defaults to `1`, 
#'   the one which induces the least amount of motion.
#'
#' @returns List with three elements:
#'   * "vl" which shows the distance (in `edo` steps) that each voice moves
#'   * "tn" which indicates the (chromatic) transposition achieved by the voice leading
#'   * "rotation" which indicates the scalar transposition caused by the voice leading
#'
#' @examples
#' major_scale <- c(0, 2, 4, 5, 7, 9, 11)
#' vlsig(major_scale) # Hook's elementary signature transformation
#'
#' pure_major_triad <- j(1, 3, 5)
#' vlsig(pure_major_triad, index=1)
#' vlsig(pure_major_triad, index=2)
#'
#' odd_pentachord <- c(0, 1, 4, 9, 11) # in 15-edo
#' vlsig(odd_pentachord, index=2, edo=15)
#' vl_rolodex(odd_pentachord, edo=15)$"8" 
#'
#' @export
vlsig <- function(set, index=1, display_digits=2, edo=12, rounder=10) {
  if (index < 1) {
    stop("Index must be positive!")
  }

  card <- length(set)
  tn_levels <- vl_generators(set, edo=edo, rounder=rounder)
  rownames(tn_levels) <- NULL

  if (index > dim(tn_levels)[2]) {
    stop(paste0(deparse(substitute(set)), " doesn't have that many VL generators!"))
  }

  chosen_tn_level <- tn_levels[2, index]
  chosen_generic_interval <- tn_levels[1, index]
  modes <- sim(set, edo=edo)

  goal_set <- rotate(set, -chosen_generic_interval) 
  goal_set[1:chosen_generic_interval] <- goal_set[1:chosen_generic_interval] - edo
  goal_set <- goal_set + chosen_tn_level

  res <- goal_set - set
  res <- round(res, display_digits)
  list(vl=res, tn=chosen_tn_level, rotation=chosen_generic_interval)
}


