#' Marshall–Olkin Generalized Inverse Weibull Distribution (MOGIW)
#'
#' @description
#' Density, distribution function, quantile function, and random generation function
#' for the MOGIW distribution with four parameters (\code{alpha}, \code{beta}, \code{lambda}, and \code{beta}).
#' See details in references (Salem, 2019).
#'
#' @param x,q    vector of quantile.
#' @param p      vector of probabilities.
#' @param n      number of observations. If \code{length(n) > 1}, the length is taken to be the number required.
#' @param alpha  scale parameter of the Generalized Inverse Weibull Distribution (GIW), where \eqn{\alpha > 0}.
#' @param beta   shape parameter#1 of the Generalized Inverse Weibull Distribution (GIW), where \eqn{\beta > 0}.
#' @param lambda shape parameter#2 of the Generalized Inverse Weibull Distribution (GIW), where \eqn{\lambda > 0}.
#' @param theta  Marshall–Olkin parameter, where \eqn{\theta > 0}.
#' @param log,log.p,log.q logical; (default = \code{FALSE}), if \code{TRUE}, then probabilities are given as \code{log(p)} or \code{log(q)}.
#' @param lower.tail logical; if \code{TRUE} (default), probabilities are \eqn{P[X \le x]}, otherwise, \eqn{P[X > x]}.
#'
#' @importFrom stats runif
#'
#' @references  Salem, H. M. (2019). The Marshall–Olkin Generalized Inverse Weibull Distribution: Properties and Application.
#' Modern Applied Science, 13(2), 54. doi:10.5539/mas.v13n2p54
#'
#' @returns
#' \code{dMOGIW} gives the density,
#' \code{pMOGIW} gives the distribution function,
#' \code{qMOGIW} gives the quantile function
#' and \code{rMOGIW} generates random samples.
#'
#' @name MOGIW
#' @export
#' @examples
#' x <- seq(0.5,4,by=0.1)
#' dMOGIW(x,1,3,2,3)
#' p<- pMOGIW(q=x,1,3,2,3)
#' q<- qMOGIW(p,1,3,2,3)
#' q
#' rMOGIW(10,1,3,2,3)
#'

#' @export
#' @rdname MOGIW
#' @examples
#' x <- seq(0.5,4,by=0.1)
#' dMOGIW(x,1,3,2,3)          #or dMOGIW(x,alpha=1,beta=3,lambda=2,theta=3)
#' dMOGIW(x,1,3,2,3,log=TRUE) #or dMOGIW(x,alpha=1,beta=3,lambda=2,theta=3,log=TRUE)
#'
dMOGIW <- function(x,alpha,beta,lambda,theta,log=FALSE){
  t <- -lambda*((alpha/x)^beta)
  E <- exp(t)
  F1 <- lambda*beta*theta*(alpha^beta)*(x^(-(beta-1)))*E
  G <- (theta-(theta-1)*E)^2
  fx<- F1/G
  if (log==FALSE)
    return (fx)
  else
    return(log(fx))
}

#' @export
#' @rdname MOGIW
#' @examples
#' q <- seq(1,4,by=0.1)
#' pMOGIW(q,1,3,2,3)   #or  pMOGIW(q,1,3,2,3,lower.tail = TRUE)
#' pMOGIW(q,1,3,2,3,lower.tail = FALSE)
#'
pMOGIW <- function(q,alpha,beta,lambda,theta,lower.tail=TRUE,log.p=FALSE){
  t <- -lambda*((alpha/q)^beta)
  E <- exp(t)
  D <- theta-(theta-1)*E
  cdf<- E/D
  if(lower.tail==TRUE)
    p <- cdf
  else
    p <- 1-cdf

  if (log.p==TRUE)
    return (log(p))
  else
    return(p)
}

#' @export
#' @rdname MOGIW
#' @examples
#' q <- seq(0.5,1.5,by=0.01)
#' p <- pMOGIW(q,1,3,2,3)
#' x <- qMOGIW(p,1,3,2,3)
#'
qMOGIW <- function(p,alpha,beta,lambda,theta,lower.tail=TRUE,log.q=FALSE){
  a<- log((1+p*theta-p)/(p*theta))
  b <- (a/lambda)^(-1/beta)
  c <- alpha*b
  if(lower.tail==TRUE)
    q <- c
  else
    q <- 1-c

  if (log.q==TRUE)
    return (log(q))
  else
    return(q)
}

#' @export
#' @rdname MOGIW
#' @examples
#' x <- rMOGIW(10,1,3,2,3)
#' x
#'
rMOGIW <- function(n,alpha,beta,lambda,theta){
  u <- runif(n)
  a<- log((1+u*theta-u)/(u*theta))
  b <- (a/lambda)^(-1/beta)
  c <- alpha*b
  return(c)
}
