% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.missoNet.R
\name{cv.missoNet}
\alias{cv.missoNet}
\title{Cross-validation for missoNet}
\usage{
cv.missoNet(
  X,
  Y,
  kfold = 5,
  rho = NULL,
  lambda.beta = NULL,
  lambda.theta = NULL,
  lambda.beta.min.ratio = NULL,
  lambda.theta.min.ratio = NULL,
  n.lambda.beta = NULL,
  n.lambda.theta = NULL,
  beta.pen.factor = NULL,
  theta.pen.factor = NULL,
  penalize.diagonal = NULL,
  beta.max.iter = 10000,
  beta.tol = 1e-05,
  theta.max.iter = 10000,
  theta.tol = 1e-05,
  eta = 0.8,
  eps = 1e-08,
  standardize = TRUE,
  standardize.response = TRUE,
  compute.1se = TRUE,
  relax.net = FALSE,
  adaptive.search = FALSE,
  shuffle = TRUE,
  seed = NULL,
  parallel = FALSE,
  cl = NULL,
  verbose = 1
)
}
\arguments{
\item{X}{Numeric matrix (\eqn{n \times p}). Predictors (no missing values).}

\item{Y}{Numeric matrix (\eqn{n \times q}). Responses. Missing values should be
coded as \code{NA}/\code{NaN}.}

\item{kfold}{Integer \eqn{\ge 2}. Number of folds (default \code{5}).}

\item{rho}{Optional numeric vector of length \eqn{q}. Working missingness
probabilities (per response). If \code{NULL} (default), estimated from \code{Y}.}

\item{lambda.beta, lambda.theta}{Optional numeric vectors. Candidate
regularization paths for \eqn{\mathbf{B}} and \eqn{\Theta}. If
\code{NULL}, sequences are generated automatically from the data.
Avoid supplying a single value because warm starts along a path are used.}

\item{lambda.beta.min.ratio, lambda.theta.min.ratio}{Optional numerics in \code{(0,1]}.
Ratio of the smallest to the largest value when generating lambda sequences
(ignored if the corresponding \code{lambda.*} is supplied).}

\item{n.lambda.beta, n.lambda.theta}{Optional integers. Lengths of the
automatically generated lambda paths (ignored if the corresponding
\code{lambda.*} is supplied).}

\item{beta.pen.factor}{Optional \eqn{p \times q} non-negative matrix of element-wise
penalty multipliers for \eqn{\mathbf{B}}. \code{Inf} = maximum penalty;
\code{0} = no penalty for the corresponding coefficient. Default: all 1s (equal penalty).}

\item{theta.pen.factor}{Optional \eqn{q \times q} non-negative matrix of element-wise
penalty multipliers for \eqn{\Theta}. Off-diagonal entries control edge
penalties; diagonal treatment is governed by \code{penalize.diagonal}. \code{Inf} =
maximum penalty; \code{0} = no penalty for that element. Default: all 1s (equal penalty).}

\item{penalize.diagonal}{Logical or \code{NULL}. Whether to penalize diagonal entries
of \eqn{\Theta}. If \code{NULL} (default) the choice is made automatically.}

\item{beta.max.iter, theta.max.iter}{Integers. Max iterations for the
\eqn{\mathbf{B}} update (FISTA) and \eqn{\Theta} update (graphical lasso).
Defaults: \code{10000}.}

\item{beta.tol, theta.tol}{Numerics \eqn{> 0}. Convergence tolerances for the
\eqn{\mathbf{B}} and \eqn{\Theta} updates. Defaults: \code{1e-5}.}

\item{eta}{Numeric in \code{(0,1)}. Backtracking line-search parameter for the
\eqn{\mathbf{B}} update (default \code{0.8}).}

\item{eps}{Numeric in \code{(0,1)}. Eigenvalue floor used to stabilize positive
definiteness operations (default \code{1e-8}).}

\item{standardize}{Logical. Standardize columns of \code{X} internally? Default \code{TRUE}.}

\item{standardize.response}{Logical. Standardize columns of \code{Y} internally?
Default \code{TRUE}.}

\item{compute.1se}{Logical. Also compute 1-SE solutions? Default \code{TRUE}.}

\item{relax.net}{(Experimental) Logical. If \code{TRUE}, refit active edges of \eqn{\Theta}
without \eqn{\ell_1} penalty (de-biased network). Default \code{FALSE}.}

\item{adaptive.search}{(Experimental) Logical. Use adaptive two-stage lambda search? Default \code{FALSE}.}

\item{shuffle}{Logical. Randomly shuffle fold assignments? Default \code{TRUE}.}

\item{seed}{Optional integer seed (used when \code{shuffle = TRUE}).}

\item{parallel}{Logical. Evaluate folds in parallel using a provided cluster?
Default \code{FALSE}.}

\item{cl}{Optional cluster from \code{parallel::makeCluster()} (required if
\code{parallel = TRUE}).}

\item{verbose}{Integer in \code{0,1,2}. \code{0} = silent, \code{1} = progress (default),
\code{2} = detailed tracing (not supported in parallel mode).}
}
\value{
A list of class \code{"missoNet"} with components:
\describe{
\item{est.min}{List of estimates at the CV minimum:
\code{Beta} (\eqn{p \times q}), \code{Theta} (\eqn{q \times q}),
intercept \code{mu} (length \eqn{q}), \code{lambda.beta}, \code{lambda.theta},
\code{lambda.beta.idx}, \code{lambda.theta.idx}, and (if requested)
\code{relax.net}.}
\item{est.1se.beta}{List of estimates at the 1-SE \code{lambda.beta}
(if \code{compute.1se = TRUE}); \code{NULL} otherwise.}
\item{est.1se.theta}{List of estimates at the 1-SE \code{lambda.theta}
(if \code{compute.1se = TRUE}); \code{NULL} otherwise.}
\item{rho}{Length-\eqn{q} vector of working missingness probabilities.}
\item{kfold}{Number of folds used.}
\item{fold.index}{Integer vector of length \eqn{n} giving fold assignments
(names are \code{"fold-k"}).}
\item{lambda.beta.seq, lambda.theta.seq}{Unique lambda values explored along
the grid for \eqn{\mathbf{B}} and \eqn{\Theta}.}
\item{penalize.diagonal}{Logical indicating whether the diagonal of
\eqn{\Theta} was penalized.}
\item{beta.pen.factor, theta.pen.factor}{Penalty factor matrices actually used.}
\item{param_set}{List with CV diagnostics:
\code{n}, \code{p}, \code{q}, \code{standardize}, \code{standardize.response},
mean errors \code{cv.errors.mean}, bounds \code{cv.errors.upper/lower},
and the evaluated grids \code{cv.grid.beta}, \code{cv.grid.theta} (length equals
number of fitted models).}
}
}
\description{
Perform \eqn{k}-fold cross-validation to select the regularization pair
\code{(lambda.beta, lambda.theta)} for \code{\link{missoNet}}. For each
fold the model is trained on \eqn{k-1} partitions and evaluated on the held-out
partition over a grid of lambda pairs; the pair with minimum mean CV error is
returned, with optional 1-SE models for more regularized solutions.
}
\details{
Internally, predictors \code{X} and responses \code{Y} can be standardized
for optimization; all reported estimates are re-scaled back to the original
data scale. Missingness in \code{Y} is handled via unbiased estimating
equations using column-wise observation probabilities estimated from \code{Y}
(or supplied via \code{rho}). This is appropriate when the missingness of each
response is independent of its unobserved value (e.g., MCAR).

If \code{adaptive.search = TRUE}, a fast two-stage pre-optimization narrows
the lambda grid before computing fold errors on a focused neighborhood; this
can be substantially faster on large grids but may occasionally miss the global
optimum.

When \code{compute.1se = TRUE}, two additional solutions are reported:
the largest \code{lambda.beta} and the largest \code{lambda.theta} whose CV
error is within one standard error of the minimum (holding the other lambda
fixed at its optimal value). At the end, three special lambda pairs are identified:
\itemize{
\item \strong{lambda.min}: Parameters giving minimum CV error
\item \strong{lambda.1se.beta}: Largest \eqn{\lambda_B} within 1 SE of minimum
(with \eqn{\lambda_\Theta} fixed at optimum)
\item \strong{lambda.1se.theta}: Largest \eqn{\lambda_\Theta} within 1 SE of minimum
(with \eqn{\lambda_B} fixed at optimum)
}

The 1SE rules provide more regularized models that may generalize better.
}
\examples{
sim <- generateData(n = 120, p = 12, q = 6, rho = 0.1)
X <- sim$X; Y <- sim$Z

\donttest{
# Basic 5-fold cross-validation
cvfit <- cv.missoNet(X = X, Y = Y, kfold = 5, verbose = 0)

# Extract optimal estimates
Beta.min <- cvfit$est.min$Beta
Theta.min <- cvfit$est.min$Theta

# Extract 1SE estimates (if computed)
if (!is.null(cvfit$est.1se.beta)) {
  Beta.1se <- cvfit$est.1se.beta$Beta
}
if (!is.null(cvfit$est.1se.theta)) {
  Theta.1se <- cvfit$est.1se.theta$Theta
}

# Make predictions
newX <- matrix(rnorm(10 * 12), 10, 12)
pred.min <- predict(cvfit, newx = newX, s = "lambda.min")
pred.1se <- predict(cvfit, newx = newX, s = "lambda.1se.beta")

# Parallel cross-validation
library(parallel)
cl <- makeCluster(min(detectCores() - 1, 2))
cvfit2 <- cv.missoNet(X = X, Y = Y, kfold = 5, 
                      parallel = TRUE, cl = cl)
stopCluster(cl)

# Adaptive search for efficiency
cvfit3 <- cv.missoNet(X = X, Y = Y, kfold = 5,
                      adaptive.search = TRUE)

# Reproducible CV with specific lambdas
cvfit4 <- cv.missoNet(X = X, Y = Y, kfold = 5,
                      lambda.beta = 10^seq(0, -2, length = 20),
                      lambda.theta = 10^seq(0, -2, length = 20),
                      seed = 486)

# Plot CV results
plot(cvfit, type = "heatmap")
plot(cvfit, type = "scatter")
}

}
\references{
Zeng, Y., et al. (2025). \emph{Multivariate regression with missing response data
for modelling regional DNA methylation QTLs}. arXiv:2507.05990.
}
\seealso{
\code{\link{missoNet}} for model fitting;
generic methods such as \code{plot()} and \code{predict()} for objects of class
\code{"missoNet"}.
}
\author{
Yixiao Zeng \email{yixiao.zeng@mail.mcgill.ca}, Celia M. T. Greenwood
}
