% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mediate_zi_vcoef.R
\name{mediate_zi_vcoef}
\alias{mediate_zi_vcoef}
\title{Mediation Sensitivity Analysis for Count and Zero-Inflated Count Data with
a Post-Treatment Confounder}
\usage{
mediate_zi_vcoef(
  model.u,
  delta.beta.u,
  model.m,
  model.y,
  sims = 1000,
  boot = FALSE,
  treat = "treat.name",
  mediator = "med.name",
  confounder = "confd.name",
  covariates = NULL,
  outcome = NULL,
  control = NULL,
  conf.level = 0.95,
  control.value = 0,
  treat.value = 1,
  long = TRUE,
  dropobs = FALSE,
  robustSE = FALSE,
  cluster = NULL,
  ...
)
}
\arguments{
\item{model.u}{A fitted model object for confounder. Can be of class 'lm',
'polr', 'bayespolr', 'glm', 'bayesglm', 'gam', 'rq', or 'survreg'.}

\item{delta.beta.u}{Sensitivity parameter as difference from the estimated
treatment coefficient in the confounder model (model.u) based on the
observed data.}

\item{model.m}{A fitted model object for mediator.  Can be of class 'lm',
'polr', 'bayespolr', 'glm', 'bayesglm', 'gam', 'rq', or 'survreg'.}

\item{model.y}{A fitted model object for outcome.  Can be of class 'lm',
'polr', 'bayespolr', 'glm', 'bayesglm', 'gam', 'vglm', 'rq', 'survreg',
or 'zeroinfl'}

\item{sims}{Number of Monte Carlo draws for nonparametric bootstrap or
quasi-Bayesian approximation.}

\item{boot}{A logical value. if 'FALSE' a quasi-Bayesian approximation is
used for confidence intervals; if 'TRUE' nonparametric bootstrap will be
used. Default is 'FALSE'.}

\item{treat}{A character string indicating the name of the treatment variable
used in the models.  The treatment can be either binary (integer or a
two-valued factor) or continuous (numeric).}

\item{mediator}{A character string indicating the name of the mediator
variable used in the models.}

\item{confounder}{A character string indicating the name of the confounder
variable used in the models.}

\item{covariates}{A list or data frame containing values for a subset of the
pre-treatment covariates in 'model.m' and 'model.y'. If provided, the
function will return the estimates conditional on those covariate values.}

\item{outcome}{A character string indicating the name of the outcome variable
in `model.y'. Only necessary if 'model.y' is of class 'survreg'; otherwise
ignored.}

\item{control}{A character string indicating the name of the control group
indicator. Only relevant if 'model.y' is of class 'gam'. If provided, 'd0',
'z0' and 'n0' are allowed to differ from 'd1', 'z1' and 'n1', respectively.}

\item{conf.level}{Level of the returned two-sided confidence intervals.
Default is to return the 2.5 and 97.5 percentiles of the simulated
quantities.}

\item{control.value}{Value of the treatment variable used as the control
condition. Default is 0.}

\item{treat.value}{Value of the treatment variable used as the treatment
condition. Default is 1.}

\item{long}{A logical value. If 'TRUE', the output will contain the entire
sets of simulation draws of the the average causal mediation effects,
direct effects, proportions mediated, and total effect. Default is 'TRUE'.}

\item{dropobs}{A logical value indicating the behavior when the model frames
of 'model.m' and 'model.y' (and the 'cluster' variable if included) are
composed of different observations. If 'TRUE', models will be re-fitted
using common data rows. If 'FALSE', error is returned. Default is 'FALSE'.}

\item{robustSE}{A logical value. If 'TRUE', heteroskedasticity-consistent
standard errors will be used in quasi-Bayesian simulations. Ignored if
'boot' is 'TRUE' or neither 'model.m' nor 'model.y' has a method for
\code{vcovHC} in the \code{sandwich} package. Default is 'FALSE'.}

\item{cluster}{A variable indicating clusters for standard errors. Note that
this should be a vector of cluster indicators itself, not a character
string for the name of the variable.}

\item{...}{other arguments passed to \code{vcovHC} in the \code{sandwich}
package: typically the 'type' argument, which is ignored if 'robustSE' is
'FALSE'. Arguments to the \code{boot} in the \code{boot} package may also
be passed, e.g. 'parallel' and 'ncpus'.}
}
\value{
\code{mediate_zi_vcoef} returns an object of class "\code{mediate}", (or
  "\code{mediate.order}" if the outcome model used is 'polr' or 'bayespolr'),
  a list that contains the components listed below.  Some of these elements
  are not available if 'long' is set to 'FALSE' by the user.

  The function \code{summary} (i.e., \code{summary.mediate},
  or \code{summary.mediate.order}) can be used to obtain a table of the
  results.  The function \code{plot} (i.e., \code{plot.mediate}, or
  \code{plot.mediate.order}) can be used to produce a plot of the estimated
  average causal mediation, average direct, and total effects along with
  their confidence intervals.

  \item{d0, d1}{point estimates for average causal mediation effects under
  the control and treatment conditions.}
  \item{d0.ci, d1.ci}{confidence intervals for average causal mediation
  effects. The confidence level is set at the value specified in
  'conf.level'.}
  \item{d0.p, d1.p}{two-sided p-values for average causal mediation effects.}
  \item{d0.sims, d1.sims}{vectors of length 'sims' containing simulation
  draws of average causal mediation effects.}
  \item{z0, z1}{point estimates for average direct effect under the control
  and treatment conditions.}
  \item{z0.ci, z1.ci}{confidence intervals for average direct effects.}
  \item{z0.p, z1.p}{two-sided p-values for average causal direct effects.}
  \item{z0.sims, z1.sims}{vectors of length 'sims' containing simulation
  draws of average direct effects.}
  \item{n0, n1}{the "proportions mediated", or the size of the average causal
  mediation effects relative to the total effect.}
  \item{n0.ci, n1.ci}{confidence intervals for the proportions mediated.}
  \item{n0.p, n1.p}{two-sided p-values for proportions mediated.}
  \item{n0.sims, n1.sims}{vectors of length 'sims' containing simulation
  draws of the proportions mediated.}
  \item{tau.coef}{point estimate for total effect.}
  \item{tau.ci}{confidence interval for total effect.}
  \item{tau.p}{two-sided p-values for total effect.}
  \item{tau.sims}{a vector of length 'sims' containing simulation draws of
  the total effect.}
  \item{d.avg, z.avg, n.avg}{simple averages of d0 and d1, z0 and z1, n0 and
  n1, respectively, which users may want to use as summary values when those
  quantities differ.}
  \item{d.avg.ci, z.avg.ci, n.avg.ci}{confidence intervals for the above.}
  \item{d.avg.p, z.avg.p, n.avg.p}{two-sided p-values for the above.}
  \item{d.avg.sims, z.avg.sims, n.avg.sims}{vectors of length 'sims'
  containing simulation draws of d.avg, z.avg and n.avg, respectively.}
  \item{boot}{logical, the 'boot' argument used. If 'FALSE' a quasi-Bayesian
  approximation was used for confidence intervals; if 'TRUE' nonparametric
  bootstrap was used}
  \item{boot.ci.type}{a character string 'perc' indicating percentile
  bootstrap confidence intervals were estimated if the argument boot = TRUE}
  \item{treat}{a character string indicating the name of the 'treat' variable
  used in the models.}
  \item{mediator}{a character string indicating the name of the 'mediator'
  variable used in the models.}
  \item{INT}{a logical value indicating whether the model specification
  allows the effects to differ between the treatment and control conditions.}
  \item{conf.level}{the confidence level used. }
  \item{model.y}{the outcome model used.}
  \item{model.m}{the mediator model used.}
  \item{control.value}{value of the treatment variable used as the control
  condition.}
  \item{treat.value}{value of the treatment variable used as the treatment
  condition.}
  \item{nobs}{number of observations in the model frame for 'model.m' and
  'model.y'. May differ from the numbers in the original models input to
  'mediate' if 'dropobs' was 'TRUE'.}
  \item{robustSE}{`TRUE' or `FALSE'.}
  \item{cluster}{the clusters used.}
}
\description{
\loadmathjax{} 'mediate_zi_vcoef' is modified from \code{mediate_zi} function with 3
confounder-related parameters ('model.u', 'delta.beta.u', and 'confounder')
added. It is used to estimate causal mediation effects when there is a
treatment-induced mediator-outcome confounder, and the coefficient of treatment
in the confounder model is specified by users. Users can perform sensitivity
analysis with a range of specified coefficient values when there is a
post-treatment confounder.
}
\examples{
data("midvd_bt100")
uFit <- glm (PDVisit_6 ~ intervention + BrushTimes_W2 + HealthyMeals_W2
                         + PDVisit_W2,
             family = 'binomial', data = midvd_bt100)
mFit <- glm (PBrushBedt_6 ~ intervention + BrushTimes_W2 + HealthyMeals_W2
                            + PBrush_W2 + PDVisit_6,
             family = 'binomial', data = midvd_bt100)
yFit <- zeroinfl(Untreated_W3 ~ intervention + PBrushBedt_6 + BrushTimes_W2
                                + HealthyMeals_W2 + PBrush_W2+ PDVisit_6,
                 data = midvd_bt100)
# For illustration purposes a small number of simulations are used
# Estimation via Quasi-Bayesian approximation
ee <-mediate_zi_vcoef(uFit, delta.beta.u = 0.01, mFit, yFit, sims = 100,
                      treat = "intervention", mediator = "PBrushBedt_6",
                      confounder ="PDVisit_6")
summary(ee)

}
\references{
Cheng, J., Cheng, N.F., Guo, Z., Gregorich, S., Ismail, A.I.,
  Gansky, S.A (2018) Mediation analysis for count and zero-inflated count
  data. Statistical Methods in Medical Research. 27(9):2756-2774.

  Ismail AI, Ondersma S, Willem Jedele JM, et al. (2011) Evaluation of
  a brief tailored motivational intervention to prevent early childhood caries.
  Community Dentistry and Oral Epidemiology 39: 433–448.

  Tingley, D., Yamamoto, T., Hirose, K., Imai, K. and Keele, L.
  (2014). "mediation: R package for Causal Mediation Analysis", Journal of
  Statistical Software, Vol. 59, No. 5, pp. 1-38.

  Imai, K., Keele, L., Tingley, D. and Yamamoto, T. (2011). Unpacking the
  Black Box of Causality: Learning about Causal Mechanisms from Experimental
  and Observational Studies, American Political Science Review, Vol. 105, No.
  4 (November), pp. 765-789.

  Imai, K., Keele, L. and Tingley, D. (2010) A General Approach to Causal
  Mediation Analysis, Psychological Methods, Vol. 15, No. 4 (December), pp.
  309-334.

  Imai, K., Keele, L. and Yamamoto, T. (2010) Identification, Inference, and
  Sensitivity Analysis for Causal Mediation Effects, Statistical Science,
  Vol. 25, No. 1 (February), pp. 51-71.

  Imai, K., Keele, L., Tingley, D. and Yamamoto, T. (2009) "Causal Mediation
  Analysis Using R" in Advances in Social Science Research Using R, ed. H. D.
  Vinod New York: Springer.
}
\seealso{
\code{\link{plot_sensitivity}}, \code{\link{mediate_zi}},
  \code{\link[mediation]{summary.mediate}}, \code{\link[mediation]{plot.mediate}}
}
\author{
Nancy Cheng,
  \email{Nancy.Cheng@ucsf.edu}; Jing Cheng,
  \email{Jing.Cheng@ucsf.edu}.
}
