% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{projection3}
\alias{projection3}
\title{Conduct Population Projection Simulations}
\usage{
projection3(
  mpm,
  nreps = 1L,
  times = 10000L,
  historical = FALSE,
  stochastic = FALSE,
  standardize = FALSE,
  growthonly = TRUE,
  integeronly = FALSE,
  substoch = 0L,
  exp_tol = 700,
  sub_warnings = TRUE,
  quiet = FALSE,
  year = NULL,
  start_vec = NULL,
  start_frame = NULL,
  tweights = NULL,
  density = NULL,
  sparse = NULL
)
}
\arguments{
\item{mpm}{A matrix projection model of class \code{lefkoMat}, or a list of
full matrix projection matrices.}

\item{nreps}{The number of replicate projections.}

\item{times}{Number of occasions to iterate per replicate. Defaults to
10,000.}

\item{historical}{An optional logical value only used if object \code{mpm}
is a list of matrices, rather than a \code{lefkoMat} object. Defaults to
\code{FALSE} for the former case, and overridden by information supplied in
the \code{lefkoMat} object for the latter case.}

\item{stochastic}{A logical value denoting whether to conduct a stochastic
projection or a deterministic / cyclical projection.}

\item{standardize}{A logical value denoting whether to re-standardize the
population size to 1.0 at each occasion. Defaults to \code{FALSE}.}

\item{growthonly}{A logical value indicating whether to produce only the
projected population size at each occasion, or a vector showing the stage
distribution followed by the reproductive value vector followed by the full
population size at each occasion. Defaults to \code{TRUE}.}

\item{integeronly}{A logical value indicating whether to round the number of
individuals projected in each stage at each occasion to the nearest
integer. Defaults to \code{FALSE}.}

\item{substoch}{An integer value indicating whether to force survival-
transition matrices to be substochastic in density dependent simulations.
Defaults to \code{0}, which does not force substochasticity. Alternatively,
\code{1} forces all survival-transition elements to range from 0.0 to 1.0,
and forces fecundity to be non-negative; and \code{2} forces all column rows
in the survival-transition matrices to total no more than 1.0, in addition
to the actions outlined for option \code{1}.}

\item{exp_tol}{A numeric value used to indicate a maximum value to set
exponents to in the core kernel to prevent numerical overflow. Defaults to
\code{700}.}

\item{sub_warnings}{A logical value indicating whether to warn the user if
density dependence yields matrix values outside of the realm of possibility.
Generally, this means that survival-transition elements altered to values
outside of the interval [0, 1], and negative fecundity values, will both
yield warnings. Defaults to \code{TRUE}, but becomes \code{FALSE} if
\code{quiet = TRUE}.}

\item{quiet}{A logical value indicating whether to suppress warnings.
Defaults to \code{FALSE}.}

\item{year}{Either a single integer value corresponding to the year to
project, or a vector of \code{times} elements with the year to use at each
time step. If a vector shorter than \code{times} is supplied, then this
vector will be cycled. If not provided, then all annual matrices will be
cycled within patches or populations.}

\item{start_vec}{An optional numeric vector denoting the starting stage
distribution for the projection. Defaults to a single individual of each
stage.}

\item{start_frame}{An optional data frame characterizing stages, age-stages,
or stage-pairs that should be set to non-zero values in the starting vector,
and what those values should be. Can only be used with \code{lefkoMat}
objects.}

\item{tweights}{An optional numeric vector or matrix denoting the
probabilities of choosing each matrix in a stochastic projection. If a
matrix is input, then a first-order Markovian environment is assumed, in
which the probability of choosing a specific annual matrix depends on which
annual matrix is currently chosen. If a vector is input, then the choice of
annual matrix is assumed to be independent of the current matrix. Defaults
to equal weighting among matrices.}

\item{density}{An optional data frame describing the matrix elements that
will be subject to density dependence, and the exact kind of density
dependence that they will be subject to. The data frame used should be an
object of class \code{lefkoDens}, which is the output from function
\code{\link{density_input}()}.}

\item{sparse}{A text string indicating whether to use sparse matrix encoding
(\code{"yes"}) or dense matrix encoding (\code{"no"}), if the
\code{lefkoMat} object input as \code{mpm} is composed of standard matrices.
Defaults to \code{"auto"}, in which case sparse matrix encoding is used with
standard, square matrices with at least 50 rows and no more than 50\% of
elements with values greater than zero, or when input \code{lefkoMat}
objects include matrices of class \code{dgCMatrix}.}
}
\value{
A list of class \code{lefkoProj}, which always includes the first
three elements of the following, and also includes the remaining elements
below when a \code{lefkoMat} object is used as input:
\item{projection}{A list of lists of matrices showing the total number of
individuals per stage per occasion. The first list corresponds to each
pop-patch followed by each population. The inner list corresponds to
replicates within each pop-patch or population.}
\item{stage_dist}{A list of lists of the actual stage distribution in each
occasion in each replicate in each pop-patch or population. The list order
is the same as in \code{projection}.}
\item{rep_value}{A list of lists of the actual reproductive value in each
occasion in each replicate in each pop-patch or population. The list order
is the same as in \code{projection}.}
\item{pop_size}{A list of matrices showing the total population size in
each occasion per replicate (row within matrix) per pop-patch or
population (list element).}
\item{labels}{A data frame showing the order of populations and patches in
item \code{projection}.}
\item{ahstages}{The original stageframe used in the study.}
\item{hstages}{A data frame showing the order of historical stage pairs.}
\item{agestages}{A data frame showing the order of age-stage pairs.}
\item{control}{A short vector indicating the number of replicates and the
number of occasions projected per replicate.}
\item{density}{The data frame input under the density option. Only provided
if input by the user.}
}
\description{
Function \code{projection3()} runs projection simulations. It projects the
population and patches forward in time by a user-defined number of
occasions. A given set of matrices is utilized and not recreated, although
elements may be altered if density dependence is set. Projections may be
deterministic or stochastic, and may be density dependent in either case. If
deterministic, then projections will be cyclical if matrices exist covering
multiple occasions for each population or patch. If stochastic, then annual
matrices will be shuffled within patches and populations. Also produces
replicates if set.
}
\section{Notes}{

Projections are run both at the patch level and at the population level.
Population level estimates will be noted at the end of the data frame with
\code{0} entries for patch designation.

Weightings given in \code{tweights} do not need to sum to 1. Final
weightings used will be based on the proportion per element of the sum of
elements in the user-supplied vector.

Starting vectors can be input in one of two ways: 1) as \code{start_vec}
input, which is a vector of numbers of the numbers of individuals in each
stage, stage pair, or age-stage, with the length of the vector necessarily
as long as there are rows in the matrices of the MPM; or 2) as
\code{start_frame} input, which is a data frame showing only those stages,
stage pairs, or age-stages that should begin with more than 0 individuals,
and the numbers of individuals that those stages should start with (this
object is created using the \code{\link{start_input}()} function). If both
are provided, then \code{start_frame} takes precedence and \code{start_vec}
is ignored. If neither is provided, then \code{projection3()} automatically
assumes that each stage, stage pair, or age-stage begins with a single
individual. Importantly, if a \code{lefkoMat} object is not used, and a list
of matrices is provided instead, then \code{start_frame} cannot be utilized
and a full \code{start_vec} must be provided to conduct a simulation with
starting numbers of individuals other than 1 per stage.

The resulting data frames in element \code{projection} are separated by
pop-patch according to the order provided in element \code{labels}, but the
matrices for each element of \code{projection} have the result of each
replicate stacked in order on top of one another without any break or
indication. Results for each replicate must be separated using the
information provided in elements \code{control} and the 3 stage
descriptor elements.

Density dependent projections are automatically set up if object
\code{density} is input. If this object is not included, then density
independent projections will be set up. Note that currently, density
dependent projections can only be performed with \code{lefkoMat} objects.

When running density dependent simulations involving user-set exponents,
such as the beta term in the Ricker function and both the alpha and beta
terms in the Usher function, values above or below the computer limits may
cause unpredictable behavior. Noted odd behavior includes sudden shifts in
population size to negative values. This function produces warnings when
such values are used, and the values used for warnings may be reset with the
\code{exp_tol} term.

The stage distributions and reproductive values produced are not the
asymptotic values as would be given by the standardized right and left
eigenvectors associated with the dominant eigenvalue of a matrix, but are
vectors describing these values at the specific points in time projected.
See equations 14.86 and 14.88 and section 14.4 on Sensitivity and Elasticity
Analysis under Environmental Stochasticity in Caswell (2001, Matrix
Population Models, Sinauer Associates) for more details.

Consistently positive population growth can quickly lead to population size
numbers larger than can be handled computationally. In that circumstance, a
continuously rising population size will suddenly become \code{NaN} for the
remainder of the projection.

Users wishing to run a projection of a single patch in a \code{lefkoMat}
object with multiple patches should subset the MPM first to contain only
the patch needed. This can be accomplished with the
\code{\link{subset_lM}()} function.

Speed can sometimes be increased by shifting from automatic sparse matrix
determination to forced dense or sparse matrix projection. This will most
likely occur when matrices have between 30 and 300 rows and columns.
Defaults work best when matrices are very small and dense, or very large and
sparse. Speed can also be maximized by keeping the default setting,
\code{integeronly = TRUE}, since the default behavior is to run each
projection (replicate) until either the end, or the population size drops
to 0. Setting \code{integeronly = FALSE} may increase runtime dramatically,
since the population size can reach extremely small levels without dropping
to 0.
}

\examples{
# Lathyrus example
data(lathyrus)

sizevector <- c(0, 100, 13, 127, 3730, 3800, 0)
stagevector <- c("Sd", "Sdl", "VSm", "Sm", "VLa", "Flo", "Dorm")
repvector <- c(0, 0, 0, 0, 0, 1, 0)
obsvector <- c(0, 1, 1, 1, 1, 1, 0)
matvector <- c(0, 0, 1, 1, 1, 1, 1)
immvector <- c(1, 1, 0, 0, 0, 0, 0)
propvector <- c(1, 0, 0, 0, 0, 0, 0)
indataset <- c(0, 1, 1, 1, 1, 1, 1)
binvec <- c(0, 100, 11, 103, 3500, 3800, 0.5)

lathframe <- sf_create(sizes = sizevector, stagenames = stagevector,
  repstatus = repvector, obsstatus = obsvector, matstatus = matvector,
  immstatus = immvector, indataset = indataset, binhalfwidth = binvec,
  propstatus = propvector)

lathvert <- verticalize3(lathyrus, noyears = 4, firstyear = 1988,
  patchidcol = "SUBPLOT", individcol = "GENET", blocksize = 9,
  juvcol = "Seedling1988", sizeacol = "Volume88", repstracol = "FCODE88",
  fecacol = "Intactseed88", deadacol = "Dead1988",
  nonobsacol = "Dormant1988", stageassign = lathframe, stagesize = "sizea",
  censorcol = "Missing1988", censorkeep = NA, censor = TRUE)

lathsupp3 <- supplemental(stage3 = c("Sd", "Sd", "Sdl", "Sdl", "Sd", "Sdl"), 
  stage2 = c("Sd", "Sd", "Sd", "Sd", "rep", "rep"),
  stage1 = c("Sd", "rep", "Sd", "rep", "all", "all"), 
  givenrate = c(0.345, 0.345, 0.054, 0.054, NA, NA),
  multiplier = c(NA, NA, NA, NA, 0.345, 0.054),
  type = c(1, 1, 1, 1, 3, 3), type_t12 = c(1, 2, 1, 2, 1, 1),
  stageframe = lathframe, historical = TRUE)

ehrlen3 <- rlefko3(data = lathvert, stageframe = lathframe,
  year = c(1989, 1990), stages = c("stage3", "stage2", "stage1"),
  supplement = lathsupp3, yearcol = "year2", indivcol = "individ")

lathproj <- projection3(ehrlen3, nreps = 5, stochastic = TRUE)

# Cypripedium example
data(cypdata)
 
sizevector <- c(0, 0, 0, 0, 0, 0, 1, 2.5, 4.5, 8, 17.5)
stagevector <- c("SD", "P1", "P2", "P3", "SL", "D", "XSm", "Sm", "Md", "Lg",
  "XLg")
repvector <- c(0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1)
obsvector <- c(0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1)
matvector <- c(0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1)
immvector <- c(0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0)
propvector <- c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0)
indataset <- c(0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1)
binvec <- c(0, 0, 0, 0, 0, 0.5, 0.5, 1, 1, 2.5, 7)

cypframe_raw <- sf_create(sizes = sizevector, stagenames = stagevector,
  repstatus = repvector, obsstatus = obsvector, matstatus = matvector, 
  propstatus = propvector, immstatus = immvector, indataset = indataset,
  binhalfwidth = binvec)

cypraw_v1 <- verticalize3(data = cypdata, noyears = 6, firstyear = 2004,
  patchidcol = "patch", individcol = "plantid", blocksize = 4, 
  sizeacol = "Inf2.04", sizebcol = "Inf.04", sizeccol = "Veg.04", 
  repstracol = "Inf.04", repstrbcol = "Inf2.04", fecacol = "Pod.04",
  stageassign = cypframe_raw, stagesize = "sizeadded", NAas0 = TRUE, 
  NRasRep = TRUE)

cypsupp3r <- supplemental(stage3 = c("SD", "SD", "P1", "P1", "P2", "P3", "SL",
    "D", "XSm", "Sm", "D", "XSm", "Sm", "mat", "mat", "mat", "SD", "P1"),
  stage2 = c("SD", "SD", "SD", "SD", "P1", "P2", "P3", "SL", "SL", "SL", "SL",
    "SL", "SL", "D", "XSm", "Sm", "rep", "rep"),
  stage1 = c("SD", "rep", "SD", "rep", "SD", "P1", "P2", "P3", "P3", "P3",
    "SL", "SL", "SL", "SL", "SL", "SL", "mat", "mat"),
  eststage3 = c(NA, NA, NA, NA, NA, NA, NA, "D", "XSm", "Sm", "D", "XSm", "Sm",
    "mat", "mat", "mat", NA, NA),
  eststage2 = c(NA, NA, NA, NA, NA, NA, NA, "XSm", "XSm", "XSm", "XSm", "XSm",
    "XSm", "D", "XSm", "Sm", NA, NA),
  eststage1 = c(NA, NA, NA, NA, NA, NA, NA, "XSm", "XSm", "XSm", "XSm", "XSm",
    "XSm", "XSm", "XSm", "XSm", NA, NA),
  givenrate = c(0.1, 0.1, 0.2, 0.2, 0.2, 0.2, 0.25, NA, NA, NA, NA, NA, NA,
    NA, NA, NA, NA, NA),
  multiplier = c(NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA,
    NA, 0.5, 0.5),
  type = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3),
  type_t12 = c(1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
  stageframe = cypframe_raw, historical = TRUE)

cypmatrix3r <- rlefko3(data = cypraw_v1, stageframe = cypframe_raw, 
  year = "all", patch = "all", stages = c("stage3", "stage2", "stage1"),
  size = c("size3added", "size2added", "size1added"), 
  supplement = cypsupp3r, yearcol = "year2", 
  patchcol = "patchid", indivcol = "individ")

cypstoch <- projection3(cypmatrix3r, nreps = 5, stochastic = TRUE)

}
\seealso{
\code{\link{start_input}()}

\code{\link{density_input}()}

\code{\link{f_projection3}()}

\code{\link{append_lP}()}

\code{\link{summary.lefkoProj}()}

\code{\link{plot.lefkoProj}()}
}
