#' Generic summary function for bcdpmeta object in jarbes
#' @param object The object generated by the bcmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param ... \dots
#'
#' @export
summary.bcdpmeta = function(object, digits = 3, ...) {
  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications ....
  #
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Hyper-priors parameters............................................
  model.spec$mean.mu     = object$prior$mean.mu
  model.spec$sd.mu       = object$prior$sd.mu
  model.spec$scale.sigma.between = object$prior$scale.sigma.between
  model.spec$df.scale.between    = object$prior$df.scale.between
  model.spec$B.lower     = object$prior$B.lower
  model.spec$B.upper     = object$prior$B.upper
  #
  model.spec$a.0         = object$prior$a.0
  model.spec$a.1         = object$prior$a.1
  model.spec$nu          = object$prior$nu
  # model.spec$nu.estimate = object$prior$nu.estimate
  # model.spec$b.0 = object$$priorb.0
  # model.spec$b.1 = object$prior$b.1

  summary.m$model.specification = model.spec

  # Posterior of the model parameters
  #
  # The list of parameters will include more complex models, e.g. estimation of
  # the parameters nu from the Beta ...
  #
  summary.m$summary.par = bugs.summary[c("mu[1]", "mu.new", "tau", "mu[2]","p.bias[2]"),]

  row.names(summary.m$summary.par) = c("Mean (corrected)",
                                       "Predictive effect (corrected)",
                                       "Tau (between studies sd)",
                                       "Mean bias",
                                       "Prob. biased class")

  # predictive effects
  # summary.m$summary.predictive.effects = bugs.summary[c("Odds.new",
  #                                                     "P_control.new"),]

  # DIC
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD

  # MCMC setup ...
  mcmc.setup = list()
  mcmc.setup$n.chains = bugs.output$n.chains
  mcmc.setup$n.iter = bugs.output$n.iter
  mcmc.setup$n.burnin = bugs.output$n.burnin
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.bcmeta"

 print(summary.m, digits, ...)
}

print.summary.bcmeta = function(x, digits, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')
  cat('  Hyper-priors parameters: \n')
  cat(paste('  Prior for mu: Normal', '[', model.spec$mean.mu,', ' ,model.spec$sd.mu^2,']', sep = ''))
  cat('\n')
  cat(paste('  Prior bias interval: Uniform', '[', model.spec$B.lower,', ' ,model.spec$B.upper,']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/tau^2: Scale.Gamma', '[', model.spec$scale.sigma.between,', ' ,
            model.spec$df.scale.between,']', sep = ''))
  cat('\n')
  cat(paste('  Prior bias probability: Beta', '[', model.spec$a.0,', ' ,model.spec$a.1,']', sep = ''))
  cat('\n')
  cat(paste('  Prior nu: ', model.spec$nu, sep = ''))
  cat('\n')


  cat('\n')
  cat('Posterior distributions: \n')
  print(round(x$summary.par, digits))

  cat('\n-------------------\n')

  #  cat('Predictive effects:\n')
  #  print(round(x$summary.predictive.effects, digits))

  #  cat('\n-------------------\n')

  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' iterations (first ', mcmc$n.burnin, ' discarded)', sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}

