#' Confint Method for 'glsm'
#'
#' @title Confidence Intervals for Coefficients in \code{glsm} Objects
#'
#' @description Calculates confidence intervals for the coefficients in a fitted \code{glsm} model. Includes exponentiated intervals (Odds Ratios) for easier interpretation.
#' @param object The type of prediction required. The default is on the scale of the linear predictors. The alternative \code{response} gives the predicted probabilities.
#' @param parm calculate confidence intervals for the coefficients
#' @param level  It gives the desired confidence level for the confidence interval. For example, a default value is level = 0.95, which will generate a 95% confidence interval."
#' The alternative \code{response} gives the predicted probabilities.
#' @param ... further arguments passed to or from other methods.
#' @return An object of class \code{"confint.glsm"}, which is a list containing:
#'
#' \item{object}{a \code{glsm} object}
#' \item{parm}{calculate confidence intervals for the coefficients.}
#' \item{level}{confidence levels}
#'
#'
#' @encoding UTF-8
#' @details The saturated model is characterized by the assumptions 1 and 2 presented in section 2.3 by Llinas (2006, ISSN:2389-8976).
#'
#' @references
#' Hosmer, D., Lemeshow, S., & Sturdivant, R. (2013). *Applied Logistic Regression* (3rd ed.). New York: Wiley. ISBN: 978-0-470-58247-3
#' Llinás, H. (2006). Precisiones en la teoría de los modelos logísticos. *Revista Colombiana de Estadística*, 29(2), 239–265.
#' Llinás, H., & Carreño, C. (2012). The Multinomial Logistic Model for the Case in Which the Response Variable Can Assume One of Three Levels and Related Models. *Revista Colombiana de Estadística*, 35(1), 131–138.
#' Orozco, E., Llinás, H., & Fonseca, J. (2020). Convergence theorems in multinomial saturated and logistic models. *Revista Colombiana de Estadística*, 43(2), 211–231.
#' Llinás, H., Arteta, M., & Tilano, J. (2016). El modelo de regresión logística para el caso en que la variable de respuesta puede asumir uno de tres niveles: estimaciones, pruebas de hipótesis y selección de modelos. *Revista de Matemática: Teoría y Aplicaciones*, 23(1), 173–197.
#'
#' @author
#' Humberto Llinás (Universidad del Norte, Barranquilla-Colombia; author),
#' Jorge Villalba (Universidad Tecnológica de Bolívar, Cartagena-Colombia; author and creator),
#' Jorge Borja (Universidad del Norte, Barranquilla-Colombia; author and creator),
#' Jorge Tilano (Universidad del Norte, Barranquilla-Colombia; author)
#'
#' @examples
#' # Load the glsm package and example dataset
#' library(glsm)
#' data("hsbdemo", package = "glsm")
#'
#' # Fit a multinomial logistic regression model using glsm()
#' model <- glsm(prog ~ ses + gender, data = hsbdemo)
#'
#' # Get confidence intervals for all model coefficients (default 95% level)
#' confint(model)
#'
#' # Get confidence intervals for a specific coefficient
#' params <- names(model$coefficients)
#'
#' results <- lapply(params, function(p) {
#'   cat("\nConfidence interval for:", p, "\n")
#'   print(confint(model, parm = p, level = 0.95))
#' })
#'
#' @exportS3Method confint glsm
#'
confint.glsm <- function(object, parm, level = 0.95, ...) {
  if (length(list(...)) > 0) {
    stop("This function does not accept additional arguments. Please review the documentation.")
  }

  # Comprobación del valor de 'level'
  if ((length(level) != 1L) || is.na(level) || (level <= 0) || (level >= 1)) {
    stop("'level' must be a single number between 0 and 1")
  }

  # Comprobación de 'parm' si está especificado
  if (!missing(parm)) {
    if (!is.character(parm)) {
      stop("`parm` must be a character string.")
    }
    if (!all(parm %in% names(object$coef))) {
      stop(paste("The coefficient", parm, "is not present in the model."))
    }
  }

  alpha <- 1 - level
  Z <- qnorm(1 - alpha / 2)

  # Calcular intervalos de confianza para todos los parámetros
  if (missing(parm)) {
    li <- object$coefficients - Z * object$Std.Error
    ls <- object$coefficients + Z * object$Std.Error

    ret <- cbind(li, ls)
    colnames(ret) <- c(paste0("lower ", 100 * (1 - level) / 2, "%"), paste0("upper ", 100 * (1 - (1 - level) / 2), "%"))

    sal <- list(confint = ret, OR = exp(ret[-1, ]), level = level * 100)
  } else {
    # Intervalos de confianza para un coeficiente específico
    if (is.na(object$Std.Error[parm])) {
      stop(paste("The standard error for the coefficient", parm, "is NA."))
    }

    li <- object$coef[parm] - Z * object$Std.Error[parm]
    ls <- object$coef[parm] + Z * object$Std.Error[parm]

    ret <- cbind(li, ls)
    colnames(ret) <- c(paste0("lower ", 100 * (1 - level) / 2, "%"), paste0("upper ", 100 * (1 - (1 - level) / 2), "%"))

    # Si el coeficiente es el intercepto, no calculamos el OR
    if (parm == "(Intercept)") {
      sal <- list(confint = ret, OR = "-", level = level * 100)
    } else {
      sal <- list(confint = ret, OR = exp(ret), level = level * 100)
    }
  }

  class(sal) <- "confint.glsm"
  return(sal)
}

#' @export
print.confint.glsm <- function(x, ...) {
  cat(x$level, "% confidence intervals for coefficients:\n", sep = "")
  print(x$confint)

  cat("\n", x$level, "% confidence intervals for Odds Ratios:\n", sep = "")
  print(x$OR)
}
