% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{rgkw}
\alias{rgkw}
\title{Generalized Kumaraswamy Distribution Random Generation}
\usage{
rgkw(n, alpha, beta, gamma, delta, lambda)
}
\arguments{
\item{n}{Number of observations. If \code{length(n) > 1}, the length is
taken to be the number required. Must be a non-negative integer.}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}
}
\value{
A vector of length \code{n} containing random deviates from the GKw
distribution. The length of the result is determined by \code{n} and the
recycling rule applied to the parameters (\code{alpha}, \code{beta},
\code{gamma}, \code{delta}, \code{lambda}). Returns \code{NaN} if parameters
are invalid (e.g., \code{alpha <= 0}, \code{beta <= 0}, \code{gamma <= 0},
\code{delta < 0}, \code{lambda <= 0}).
}
\description{
Generates random deviates from the five-parameter Generalized Kumaraswamy (GKw)
distribution defined on the interval (0, 1).
}
\details{
The generation method relies on the transformation property: if
\eqn{V \sim \mathrm{Beta}(\gamma, \delta+1)}, then the random variable \code{X}
defined as
\deqn{
X = \left\{ 1 - \left[ 1 - V^{1/\lambda} \right]^{1/\beta} \right\}^{1/\alpha}
}
follows the GKw(\eqn{\alpha, \beta, \gamma, \delta, \lambda}) distribution.

The algorithm proceeds as follows:
\enumerate{
\item Generate \code{V} from \code{stats::rbeta(n, shape1 = gamma, shape2 = delta + 1)}.
\item Calculate \eqn{v = V^{1/\lambda}}.
\item Calculate \eqn{w = (1 - v)^{1/\beta}}.
\item Calculate \eqn{x = (1 - w)^{1/\alpha}}.
}
Parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}, \code{lambda})
are recycled to match the length required by \code{n}. Numerical stability is
maintained by handling potential edge cases during the transformations.
}
\examples{
\donttest{
set.seed(1234) # for reproducibility

# Generate 1000 random values from a specific GKw distribution (Kw case)
x_sample <- rgkw(1000, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
summary(x_sample)

# Histogram of generated values compared to theoretical density
hist(x_sample, breaks = 30, freq = FALSE, # freq=FALSE for density scale
     main = "Histogram of GKw(2,3,1,0,1) Sample", xlab = "x", ylim = c(0, 2.5))
curve(dgkw(x, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1),
      add = TRUE, col = "red", lwd = 2, n = 201)
legend("topright", legend = "Theoretical PDF", col = "red", lwd = 2, bty = "n")

# Comparing empirical and theoretical quantiles (Q-Q plot)
prob_points <- seq(0.01, 0.99, by = 0.01)
theo_quantiles <- qgkw(prob_points, alpha = 2, beta = 3, gamma = 1, delta = 0, lambda = 1)
emp_quantiles <- quantile(x_sample, prob_points)

plot(theo_quantiles, emp_quantiles, pch = 16, cex = 0.8,
     main = "Q-Q Plot for GKw(2,3,1,0,1)",
     xlab = "Theoretical Quantiles", ylab = "Empirical Quantiles (n=1000)")
abline(a = 0, b = 1, col = "blue", lty = 2)

# Using vectorized parameters: generate 1 value for each alpha
alphas_vec <- c(0.5, 1.0, 2.0)
n_param <- length(alphas_vec)
samples_vec <- rgkw(n_param, alpha = alphas_vec, beta = 2, gamma = 1, delta = 0, lambda = 1)
print(samples_vec) # One sample for each alpha value
# Result length matches n=3, parameters alpha recycled accordingly

# Example with invalid parameters (should produce NaN)
invalid_sample <- rgkw(1, alpha = -1, beta = 2, gamma = 1, delta = 0, lambda = 1)
print(invalid_sample)
}
}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation}

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{dgkw}}, \code{\link{pgkw}}, \code{\link{qgkw}},
\code{\link[stats]{rbeta}}, \code{\link[base]{set.seed}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{random}
