% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hsekw}
\alias{hsekw}
\title{Hessian Matrix of the Negative Log-Likelihood for the EKw Distribution}
\usage{
hsekw(par, data)
}
\arguments{
\item{par}{A numeric vector of length 3 containing the distribution parameters
in the order: \code{alpha} (\eqn{\alpha > 0}), \code{beta} (\eqn{\beta > 0}),
\code{lambda} (\eqn{\lambda > 0}).}

\item{data}{A numeric vector of observations. All values must be strictly
between 0 and 1 (exclusive).}
}
\value{
Returns a 3x3 numeric matrix representing the Hessian matrix of the
negative log-likelihood function, \eqn{-\partial^2 \ell / (\partial \theta_i \partial \theta_j)},
where \eqn{\theta = (\alpha, \beta, \lambda)}.
Returns a 3x3 matrix populated with \code{NaN} if any parameter values are
invalid according to their constraints, or if any value in \code{data} is
not in the interval (0, 1).
}
\description{
Computes the analytic 3x3 Hessian matrix (matrix of second partial derivatives)
of the negative log-likelihood function for the Exponentiated Kumaraswamy (EKw)
distribution with parameters \code{alpha} (\eqn{\alpha}), \code{beta}
(\eqn{\beta}), and \code{lambda} (\eqn{\lambda}). This distribution is the
special case of the Generalized Kumaraswamy (GKw) distribution where
\eqn{\gamma = 1} and \eqn{\delta = 0}. The Hessian is useful for estimating
standard errors and in optimization algorithms.
}
\details{
This function calculates the analytic second partial derivatives of the
negative log-likelihood function based on the EKw log-likelihood
(\eqn{\gamma=1, \delta=0} case of GKw, see \code{\link{llekw}}):
\deqn{
\ell(\theta | \mathbf{x}) = n[\ln(\lambda) + \ln(\alpha) + \ln(\beta)]
+ \sum_{i=1}^{n} [(\alpha-1)\ln(x_i) + (\beta-1)\ln(v_i) + (\lambda-1)\ln(w_i)]
}
where \eqn{\theta = (\alpha, \beta, \lambda)} and intermediate terms are:
\itemize{
\item \eqn{v_i = 1 - x_i^{\alpha}}
\item \eqn{w_i = 1 - v_i^{\beta} = 1 - (1-x_i^{\alpha})^{\beta}}
}
The Hessian matrix returned contains the elements \eqn{- \frac{\partial^2 \ell(\theta | \mathbf{x})}{\partial \theta_i \partial \theta_j}}
for \eqn{\theta_i, \theta_j \in \{\alpha, \beta, \lambda\}}.

Key properties of the returned matrix:
\itemize{
\item Dimensions: 3x3.
\item Symmetry: The matrix is symmetric.
\item Ordering: Rows and columns correspond to the parameters in the order
\eqn{\alpha, \beta, \lambda}.
\item Content: Analytic second derivatives of the \emph{negative} log-likelihood.
}
This corresponds to the relevant 3x3 submatrix of the 5x5 GKw Hessian (\code{\link{hsgkw}})
evaluated at \eqn{\gamma=1, \delta=0}. The exact analytical formulas are implemented directly.
}
\examples{
\donttest{
# Assuming existence of rekw, llekw, grekw, hsekw functions for EKw

# Generate sample data
set.seed(123)
true_par_ekw <- c(alpha = 2, beta = 3, lambda = 0.5)
if (exists("rekw")) {
  sample_data_ekw <- rekw(100, alpha = true_par_ekw[1], beta = true_par_ekw[2],
                          lambda = true_par_ekw[3])
} else {
  sample_data_ekw <- rgkw(100, alpha = true_par_ekw[1], beta = true_par_ekw[2],
                         gamma = 1, delta = 0, lambda = true_par_ekw[3])
}
hist(sample_data_ekw, breaks = 20, main = "EKw(2, 3, 0.5) Sample")

# --- Find MLE estimates ---
start_par_ekw <- c(1.5, 2.5, 0.8)
mle_result_ekw <- stats::optim(par = start_par_ekw,
                               fn = llekw,
                               gr = if (exists("grekw")) grekw else NULL,
                               method = "BFGS",
                               hessian = TRUE, # Ask optim for numerical Hessian
                               data = sample_data_ekw)

# --- Compare analytical Hessian to numerical Hessian ---
if (mle_result_ekw$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE) &&
    exists("hsekw")) {

  mle_par_ekw <- mle_result_ekw$par
  cat("\nComparing Hessians for EKw at MLE estimates:\n")

  # Numerical Hessian of llekw
  num_hess_ekw <- numDeriv::hessian(func = llekw, x = mle_par_ekw, data = sample_data_ekw)

  # Analytical Hessian from hsekw
  ana_hess_ekw <- hsekw(par = mle_par_ekw, data = sample_data_ekw)

  cat("Numerical Hessian (EKw):\n")
  print(round(num_hess_ekw, 4))
  cat("Analytical Hessian (EKw):\n")
  print(round(ana_hess_ekw, 4))

  # Check differences
  cat("Max absolute difference between EKw Hessians:\n")
  print(max(abs(num_hess_ekw - ana_hess_ekw)))

  # Optional: Use analytical Hessian for Standard Errors
  # tryCatch({
  #   cov_matrix_ekw <- solve(ana_hess_ekw)
  #   std_errors_ekw <- sqrt(diag(cov_matrix_ekw))
  #   cat("Std. Errors from Analytical EKw Hessian:\n")
  #   print(std_errors_ekw)
  # }, error = function(e) {
  #   warning("Could not invert analytical EKw Hessian: ", e$message)
  # })

} else {
  cat("\nSkipping EKw Hessian comparison.\n")
  cat("Requires convergence, 'numDeriv' package, and function 'hsekw'.\n")
}

}

}
\references{
Nadarajah, S., Cordeiro, G. M., & Ortega, E. M. (2012). The exponentiated
Kumaraswamy distribution. \emph{Journal of the Franklin Institute}, \emph{349}(3),

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

(Note: Specific Hessian formulas might be derived or sourced from additional references).
}
\seealso{
\code{\link{hsgkw}} (parent distribution Hessian),
\code{\link{llekw}} (negative log-likelihood for EKw),
\code{grekw} (gradient for EKw, if available),
\code{\link{dekw}} (density for EKw),
\code{\link[stats]{optim}},
\code{\link[numDeriv]{hessian}} (for numerical Hessian comparison).
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{hessian}
\keyword{likelihood}
\keyword{optimize}
