#' Add two triangular fuzzy numbers
#'
#' Performs the addition of two triangular fuzzy numbers.
#'
#' @param X List. First triangular fuzzy number with components `l`, `x`, and `r`.
#' @param Y List. Second triangular fuzzy number with components `l`, `x`, and `r`.
#' @return A list representing the sum of the two fuzzy numbers.
#' @examples
#' X <- list(l = 1, x = 2, r = 3)
#' Y <- list(l = 2, x = 3, r = 4)
#' fuzzy_add(X, Y)
#' @export
fuzzy_add <- function(X, Y) {
  return(list(l = X$l + Y$l, x = X$x + Y$x, r = X$r + Y$r))
}

#' Multiply two triangular fuzzy numbers
#'
#' Computes the scalar product of two triangular fuzzy numbers.
#'
#' @param X List. First triangular fuzzy number with components `l`, `x`, and `r`.
#' @param Y List. Second triangular fuzzy number with components `l`, `x`, and `r`.
#' @return A scalar representing the sum of the product of the corresponding components.
#' @examples
#' X <- list(l = 1, x = 2, r = 3)
#' Y <- list(l = 2, x = 3, r = 4)
#' fuzzy_mults(X, Y)
#' @export
fuzzy_mults <- function(X, Y) {
  return(X$l * Y$l + X$x * Y$x + X$r * Y$r)
}

#' Multiply a crisp scalar by a triangular fuzzy number
#'
#' Scales a triangular fuzzy number by a crisp scalar.
#'
#' @param scalar Numeric. The scalar to multiply with the fuzzy number.
#' @param fuzzy_num List. A triangular fuzzy number with components `l`, `x`, and `r`.
#' @return A list representing the scaled fuzzy number.
#' @examples
#' scalar <- 3
#' fuzzy_num <- list(l = 1, x = 2, r = 3)
#' fuzzy_crisp_mult(scalar, fuzzy_num)
#' @export
fuzzy_crisp_mult <- function(scalar, fuzzy_num) {
  return(list(
    l = scalar * fuzzy_num$l,
    x = scalar * fuzzy_num$x,
    r = scalar * fuzzy_num$r
  ))
}

#' Compute the squared distance between two fuzzy numbers
#'
#' Calculates the squared distance between two triangular fuzzy numbers using Diamond's metric.
#'
#' @param X List. First triangular fuzzy number.
#' @param Y List. Second triangular fuzzy number.
#' @return Numeric. The squared distance between `X` and `Y`.
#' @examples
#' X <- list(l = 1, x = 2, r = 3)
#' Y <- list(l = 2, x = 3, r = 4)
#' fuzzy_d_squared(X, Y)
#' @export
fuzzy_d_squared <- function(X, Y) {
  if (!is.list(X) || !is.list(Y)) {
    stop("Both X and Y must be lists representing triangular fuzzy numbers.")
  }

  left_dist_sq <- (Y$l - X$l)^2  # Left side distance (lower bounds)
  center_dist_sq <- (Y$x - X$x)^2  # Center value distance (modes)
  right_dist_sq <- (Y$r - X$r)^2  # Right side distance (upper bounds)

  return(left_dist_sq + center_dist_sq + right_dist_sq)
}
