#' Fuzzify a single crisp value
#'
#' Converts a crisp value into a triangular fuzzy number with a specified spread.
#'
#' @param crisp_value Numeric. The crisp value to be fuzzified.
#' @param spread Numeric. The spread for fuzzification (default is 1).
#' @return A list representing the triangular fuzzy number with components `l`, `x`, and `r`.
#' @examples
#' fuzzify_crisp_value(10, spread = 2)
#' @export
fuzzify_crisp_value <- function(crisp_value, spread = 1) {
  if (!is.numeric(crisp_value) || length(crisp_value) != 1) {
    stop("crisp_value must be a single numeric value.")
  }
  if (!is.numeric(spread) || spread < 0) {
    stop("spread must be a non-negative numeric value.")
  }

  l <- crisp_value - spread
  x <- crisp_value
  r <- crisp_value + spread
  return(list(l = l, x = x, r = r))
}

#' Fuzzify a vector of crisp values
#'
#' Converts a numeric vector into a list of fuzzified values using a triangular fuzzy membership function.
#'
#' @param crisp_vector A numeric vector to be fuzzified.
#' @param spread A non-negative numeric value specifying the spread for the fuzzy membership function.
#' @param var_name Optional. A character string specifying a common name for all fuzzified values. Default is \code{NULL}.
#'
#' @return A list of fuzzified values, where each value is represented as a list with components \code{l}, \code{x}, and \code{r}.
#' @examples
#' crisp_vector <- c(10, 20, 30)
#' fuzzify_crisp_vector(crisp_vector, spread = 1, var_name = "Variable")
#' @export
fuzzify_crisp_vector <- function(crisp_vector, spread = 1, var_name = "Outcome") {
  # Error checks
  if (!is.numeric(crisp_vector)) stop("crisp_vector must be numeric.")
  if (!is.numeric(spread) || spread < 0) stop("spread must be a non-negative numeric value.")

  # Fuzzify each value using the existing fuzzify_crisp_value function
  fuzzified_values <- lapply(crisp_vector, function(value) {
    fuzzify_crisp_value(value, spread)
  })

  # Assign unique names to each element in the list
  names(fuzzified_values) <- paste0(var_name, "_", seq_along(crisp_vector))

  return(fuzzified_values)
}

# fuzzify_crisp_vector <- function(crisp_vector, spread = 1) {
#   if (!is.numeric(crisp_vector)) {
#     stop("crisp_vector must be numeric.")
#   }
#   if (!is.numeric(spread) || spread < 0) {
#     stop("spread must be a non-negative numeric value.")
#   }
#
#   return(lapply(crisp_vector, function(crisp_value) fuzzify_crisp_value(crisp_value, spread)))
# }

#' Fuzzify a matrix of crisp values
#'
#' Converts a numeric matrix into a list of triangular fuzzy numbers.
#'
#' @param crisp_matrix Numeric matrix to be fuzzified.
#' @param spread Numeric. The spread for fuzzification (default is 1).
#' @return A list of lists representing rows of triangular fuzzy numbers.
#' @examples
#' set.seed(123)
#' matrix <- matrix(runif(9, 5, 15), nrow = 3, ncol = 3)
#' fuzzify_crisp_matrix(matrix, spread = 1.5)
#' @export
fuzzify_crisp_matrix <- function(crisp_matrix, spread = 1) {
  if (!is.matrix(crisp_matrix)) {
    stop("crisp_matrix must be a matrix.")
  }
  if (!is.numeric(spread) || spread < 0) {
    stop("spread must be a non-negative numeric value.")
  }

  return(lapply(1:nrow(crisp_matrix), function(i) {
    lapply(crisp_matrix[i, ], function(crisp_value) fuzzify_crisp_value(crisp_value, spread))
  }))
}

# fuzzify_crisp_matrix <- function(crisp_matrix, spread = 1) {
#   if (!is.matrix(crisp_matrix)) {
#     stop("crisp_matrix must be a matrix.")
#   }
#   if (!is.numeric(spread) || spread < 0) {
#     stop("spread must be a non-negative numeric value.")
#   }
#
#   return(lapply(1:nrow(crisp_matrix), function(i) {
#     lapply(crisp_matrix[i, ], function(crisp_value) fuzzify_crisp_value(crisp_value, spread))
#   }))
# }
