#' Extract Model Coefficients
#'
#' Extract estimated model parameters from objects returned by [fcm()].
#' Optionally computes confidence intervals via either the observed Hessian
#' (Delta method, on the log scale) or bootstrap sampling.
#'
#' If `method = "hessian"`, confidence intervals are constructed on the log scale
#' using the Delta method, then exponentiated to return to the original parameter scale.
#' If `method = "boot"`, confidence intervals are computed as empirical quantiles of
#' the bootstrap replicates.
#'
#' @param object An object of class `"fcm"`, typically the result of [fcm()].
#' @param ... Further arguments passed to or from other methods.
#' @param ci Confidence level for the interval estimation (e.g., 0.95). If `NULL`,
#' no confidence interval is returned.
#' @param method Character string specifying the method used to compute confidence intervals.
#' One of `"default"` (point estimate only), `"hessian"` (Delta method on log-scale),
#' or `"boot"` (percentile bootstrap).
#'
#' @returns
#' If `method = "default"`, returns a named vector of parameter estimates.
#' If `method = "hessian"` or `method = "boot"`, returns a `data.frame` with columns:
#' - `par`: the estimated parameter
#' - `lower`: lower bound of the CI
#' - `upper`: upper bound of the CI
#'
#' @examples
#' \donttest{
#' data(fit)
#' coef(fit)
#' coef(fit, method = "hessian", ci = 0.95)
#' coef(fit, method = "boot", ci = 0.95)
#' }
#'
#' @seealso [fcm()]
#' @method coef fcm
#' @export
#'
#' @srrstats {G1.5} *Software should include all code necessary to reproduce results which form the basis of performance claims made in associated publications.*
#' @srrstats {G1.6} *Software should include code necessary to compare performance claims with alternative implementations in other R packages.*
#' @srrstats {G2.1} *Implement assertions on types of inputs (see the initial point on nomenclature above).*
#' @srrstats {G2.2} *Appropriately prohibit or restrict submission of multivariate input to parameters expected to be univariate.*
#' @srrstats {G2.3a} *Use `match.arg()` or equivalent where applicable to only permit expected values.*
#' @srrstats {G2.3b} *Ensure case-insensitive handling of character arguments or document case-sensitivity.*
#' @srrstats {G2.6} *Handle all common univariate input types robustly.*
#' @srrstats {G2.10} *Column extraction should behave consistently regardless of input data class.*
#' @srrstats {G3.1} *Support multiple methods (Hessian and bootstrap) for CI estimation.*
coef.fcm <- function(object, ..., method = c("default", "hessian", "boot"), ci = 0.95) {
  stopifnot(inherits(object, "fcm"))
  stopifnot(is.numeric(object$par), length(ci) == 1, is.numeric(ci))

  method <- match.arg(method)
  par <- object$par

  if (method == "default") {
    return(par)
  }

  if (method == "boot") {
    if (is.null(object$boot)) {
      warning("Bootstrap results not available.")
      return(par)
    }
    lower <- apply(object$boot, 2, quantile, (1 - ci) / 2, na.rm = TRUE)
    upper <- apply(object$boot, 2, quantile, 1 - (1 - ci) / 2, na.rm = TRUE)
    return(data.frame(par = par, lower = lower, upper = upper))
  }

  if (method == "hessian") {
    par_log <- log(par)
    se <- sqrt(diag(solve(object$hessian)))
    z <- qnorm(1 - (1 - ci) / 2)
    lower <- par_log - z * se
    upper <- par_log + z * se
    return(data.frame(par = par, lower = exp(lower), upper = exp(upper)))
  }
}
