#' Format distance or similarity matrix output
#'
#' Converts a distance matrix to either a similarity matrix or a `dist` object,
#' depending on user preferences.
#'
#' When converting to similarity, two transformation formulas are supported to derive
#' similarity from distance:
#'
#' \describe{
#'   \item{\code{"linear", (default)}}{
#'     \deqn{\text{s}_{ij} = 1 - \delta_{ij}}
#'     This transformation directly inverts the distance into a similarity score.
#'   }
#'   \item{\code{"sqrt"}}{
#'     \deqn{\text{s}_{ij} = 1 - \delta_{ij}^2}
#'     This corresponds to a transformation from a metric that satisfies the Euclidean property:
#'     \deqn{\delta_{ij} = \sqrt{1 - s_{ij}}}
#'     According to \insertCite{gower1986metric}{dbrobust}, this transformation yields a metric that is
#'     more likely to preserve Euclidean structure in downstream analyses.
#'   }
#' }
#'
#' @param dist_mat A symmetric matrix of pairwise distances.
#' @param output_format Character string specifying output format:
#'   \code{"matrix"}, \code{"dist"}, or \code{"similarity"}.
#' @param similarity Logical; if \code{TRUE}, converts distances to similarities.
#' @param similarity_transform Character string; either \code{"linear"} (default) or \code{"sqrt"}.
#'
#' @return A matrix or `dist` object, depending on the selected format and similarity flag.
#'
#' @keywords internal
#' @references
#' \insertRef{gower1986metric}{dbrobust}
#'
#' @importFrom Rdpack reprompt
format_output <- function(dist_mat, output_format, similarity = FALSE, similarity_transform = "linear") {
  if (similarity) {
    # Choose similarity conversion formula
    if (similarity_transform == "linear") {
      sim_mat <- 1 - dist_mat
    } else if (similarity_transform == "sqrt") {
      sim_mat <- 1 - dist_mat^2
    } else {
      stop("Invalid similarity_transform option. Use 'linear' or 'sqrt'.")
    }
    return(sim_mat)
  }

  # Return distance matrix in requested format
  if (output_format == "dist") {
    return(as.dist(dist_mat))
  } else if (output_format == "matrix") {
    return(dist_mat)
  } else {
    stop("Invalid output_format: choose 'dist', 'matrix', or 'similarity'")
  }
}
