% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calibrate.R
\name{calibrate}
\alias{calibrate}
\alias{calibrate,EffectScenario-method}
\alias{calibrate,CalibrationSet-method}
\alias{calibrate,list-method}
\alias{calibrate,ScenarioSequence-method}
\title{Fit model parameters to experimental data}
\usage{
\S4method{calibrate}{EffectScenario}(x, output, data, ...)

\S4method{calibrate}{list}(
  x,
  par,
  output,
  err_fun = "sse",
  log_scale = FALSE,
  verbose = TRUE,
  ode_method = NULL,
  ...
)
}
\arguments{
\item{x}{either a single \link{scenario}. \link{sequence}, or a list of \link{caliset} objects
to be fitted. If only a scenario or sequence is supplied, the additional argument
\code{data} is required.}

\item{output}{\code{character}, name of a single output column of \code{\link[=simulate]{simulate()}} to
optimize on}

\item{data}{a \code{data.frame} or return value of \code{\link[=tox_data]{tox_data()}}; the scenarios's
output is fitted to the (observed) data. See \code{\link[=tox_data]{tox_data()}} for valid tabular
data formats.}

\item{...}{additional parameters passed on to \code{\link[stats:optim]{stats::optim()}} and \code{\link[=simulate]{simulate()}}}

\item{par}{named numeric vector with parameters to fit and their start values}

\item{err_fun}{a \code{character} choosing one of the pre-defined error functions,
or alternatively a function implementing a custom error function. Defaults
to \emph{Sum of squared errors} (\code{"sse"}).}

\item{log_scale}{logical, if \code{TRUE} then observed and predicted values are
log-transformed before the error function is evaluated. Defaults to \code{FALSE}.}

\item{verbose}{\code{logical}, if \code{TRUE} then debug outputs are displayed during
optimization}

\item{ode_method}{optional \code{character} to select an ODE solver for \code{\link[=simulate]{simulate()}}}
}
\value{
A list of fitted parameters (as produced by \code{\link[stats:optim]{stats::optim()}})
is returned.
}
\description{
The function \code{calibrate()} performs the calibration (fitting) of model
parameters to observed data. The data can originate from one or more experiments or
trials. Experimental conditions, such as model parameters and exposure
level, can differ between trials; fitting can be performed on all datasets
at the same time.
}
\details{
Fitting of model parameters can be performed in two ways:
\enumerate{
\item A single \link{scenario} is fitted to a single dataset.
The dataset must represent a time-series of an output variable of the
model, e.g. observed biomass over time (effect data). The dataset can represent
results of one or more experimental replicates under identical conditions.
\item One or more datasets of observed data are fitted each to a scenario which
describes the experimental conditions during observation, such as exposure
level and environmental properties. Each combination of dataset and scenario
is represented by a \link[=caliset]{calibration set}. During fitting,
all \emph{calibration sets} are evaluated and a total error term is calculated
over all observed and predicted values.
}
\subsection{Observed data}{

Experimental, or effect, data must be supplied as a \code{data.frame} in long format
with at least two columns: the first column contains \code{numeric} timestamps and
the remaining columns must contain the observed quantity. The observed quantitiy
must match in unit and meaning with the model output defined by argument \code{output}.

As an example, the simulation result of \link{Lemna_Schmitt} model contains the
output column \emph{biomass} (\code{BM}), amongst others. To fit model parameters of said
\emph{Lemna_Schmitt} scenario based on observed biomass, the observed data must
represent biomass as well. A minimal observed dataset could look like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{observed <- data.frame(time=c(0,  7, 14, 21),
                       BM=c( 12, 23, 37, 56))
}\if{html}{\out{</div>}}
}

\subsection{Error function}{

The error function quantifies the deviations between simulated and observed data.
The decision for an error function can have influence on the result of the
fitting procedure. Two error functions are pre-defined by the package and
can be selected by the user, but custom error functions can be used as well.

Available pre-defined error functions:
\itemize{
\item \code{"sse"}: Sum of squared errors
}

By default, the sum of squared errors is used as the error function which
gets minimized during fitting. A custom error function must accept four vectorized
arguments and return a numeric of length one, i.e. the total error value.
The arguments to the error function will all have the same length and are
defined as follows:
\itemize{
\item First argument: all observed data points
\item Second argument: all simulated data points
\item Third argument: optional weights for each data point
\item Fourth argument: a list of optional caliset tags
}

You can choose to ignore certain arguments, such as weights and tags, in your
custom error function. An example of a custom error function which returns
the sum of absolute errors looks as follow:

\if{html}{\out{<div class="sourceCode">}}\preformatted{my_absolute_error <- function(observed, predicted, weights, tags) \{
  sum(abs(observed - predicted))
\}
}\if{html}{\out{</div>}}

As tags are optional, the fourth argument may be a list containing \code{NULL} values.
The fourth argument can be used to pass additional information to the error
function: For example, the tag may identify the study from where the data
originates from and the error function could group and evaluate the data
accordingly.
}
}
\section{Methods (by class)}{
\itemize{
\item \code{calibrate(EffectScenario)}: Fit single \link{scenario} to a (\link{tox_data}) dataset

}}
\examples{
# Create an artificial data set of observed frond numbers.
# It assumes exponential growth with an effective growth rate of 0.38
trial <- data.frame(time=0:14,
                   fronds=12 * exp(0:14 * 0.38))
plot(trial)

# Create a Lemna scenario that represents unrestricted, exponential growth.
scenario <- Lemna_Schmitt() \%>\%
  set_param(c(k_phot_max=1, k_resp=0, EC50=1, b=1, P_up=1)) \%>\%
  set_init(c(BM=12)) \%>\%
  set_noexposure()

# Fit scenario parameter 'k_phot_max' to observed frond numbers:
fit <- calibrate(
  scenario,
  par="k_phot_max",
  data=trial,
  output="BM"
)

# The fitted value of 'k_phot_max' matches the effective growth rate which
# was used to create the artificial data set:
fit$par
}
