#' Get Product Use Categories
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing product use categories.
#' @export
#'
#' @examplesIf FALSE
#' # Get product use categories for Caffeine
#' get_product_use_category('DTXSID0020232')
get_product_use_category <- function(DTXSID = NULL,
                                     API_key = NULL,
                                     Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                     verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/ccd/puc/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get Production Volume
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing production volume data.
#' @export
#'
#' @examplesIf FALSE
#' # Get production volume data for Caffeine
#' get_production_volume('DTXSID0020232')
get_production_volume <- function(DTXSID = NULL,
                                  API_key = NULL,
                                  Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                  verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/ccd/production-volume/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get Biomonitoring data
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Projection Optional parameter controlling return type.
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing general use keywords.
#' @export
#'
#' @examplesIf FALSE
#' # Get biomonitoring data for Caffeine
#' get_biomonitoring_data('DTXSID0020232')
get_biomonitoring_data <- function(DTXSID = NULL,
                                   API_key = NULL,
                                   Projection = '',
                                   Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                   verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  if (!is.character(Projection)){
    warning('Setting `Projection` to ""')
    Projection <- ''
  }

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }


  response <- httr::GET(url = paste0(Server, '/ccd/monitoring-data/search/by-dtxsid/', DTXSID, ifelse(Projection == '', '', paste0('?projection=',gsub(' ', '+', Projection)))),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )

  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get General Use Keywords
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing general use keywords.
#' @export
#'
#' @examplesIf FALSE
#' # Get general use keywords for Caffeine
#' get_general_use_keywords('DTXSID0020232')
get_general_use_keywords <- function(DTXSID = NULL,
                                     API_key = NULL,
                                     Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                     verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/ccd/keywords/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get Reported Functional Use
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing reported functional use data.
#' @export
#'
#' @examplesIf FALSE
#' # Get reported functional use data for Caffeine
#' get_reported_functional_use('DTXSID0020232')
get_reported_functional_use <- function(DTXSID = NULL,
                                        API_key = NULL,
                                        Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                        verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/ccd/functional-use/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get Chemical Weight Fractions
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame containing chemical weight fraction data.
#' @export
#'
#' @examplesIf FALSE
#' # Get chemical weight fraction data for Caffeine
#' get_chemical_weight_fraction('DTXSID0020232')
get_chemical_weight_fraction <- function(DTXSID = NULL,
                                         API_key = NULL,
                                         Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                         verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/ccd/chem-weight-fractions/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}



#' Retrieve exposure related functional use data
#'
#' @param DTXSID Chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame of functional use data.
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull functional use data for BPA
#' bpa <- get_exposure_functional_use(DTXSID = 'DTXSID7020182')

get_exposure_functional_use <- function(DTXSID = NULL,
                                        API_key = NULL,
                                        Server = exposure_api_server,
                                        verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/functional-use/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}


#' Retrieve probability of exposure for functional use category
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame with probabilities corresponding to various routes of
#'   exposure related to functional use.
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull functional use probability data for BPA
#' bpa <- get_exposure_functional_use_probability(DTXSID = 'DTXSID7020182')

get_exposure_functional_use_probability <- function(DTXSID = NULL,
                                                    API_key = NULL,
                                                    Server = exposure_api_server,
                                                    verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/functional-use/probability/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Retrieve functional use categories
#'
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame of functional use categories.
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull functional use category data for BPA
#' functional_use_categories <- get_exposure_functional_use_category()

get_exposure_functional_use_category <- function(API_key = NULL,
                                                 Server = exposure_api_server,
                                                 verbose = FALSE){
  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/functional-use/category'),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get httk data
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @return A data.table of httk data for the given input chemical.
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull httk data for BPA
#' bpa_httk <- get_httk_data(DTXSID = 'DTXSID7020182')

get_httk_data <- function(DTXSID = NULL,
                          API_key = NULL,
                          Server = exposure_api_server,
                          verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/httk/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
   if(response$status_code == 401){
     stop(httr::content(response)$detail)
   }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}


#' Retrieve product data for exposure purposes
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame with product information relating to exposure to the
#'   given chemical
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull exposure product data for BPA
#' bpa <- get_exposure_product_data(DTXSID = 'DTXSID7020182')

get_exposure_product_data <- function(DTXSID = NULL,
                                      API_key = NULL,
                                      Server = exposure_api_server,
                                      verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/product-data/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Retrieve product use categories related to exposure
#'
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame consisting of all the product use categories
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull product data use categories for BPA
#' puc_categories <- get_exposure_product_data_puc()

get_exposure_product_data_puc <- function(API_key = NULL,
                                          Server = exposure_api_server,
                                          verbose = FALSE){
  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/product-data/puc'),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
                        )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}


#' Retrieve list presence tags
#'
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame with all the list presence tags and associated data.
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull list presence tags
#' tags <- get_exposure_list_presence_tags()

get_exposure_list_presence_tags <- function(API_key = NULL,
                                            Server = exposure_api_server,
                                            verbose = FALSE){
  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/list-presence/tags'),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
                        )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Retrieve document data and list presence tags for a chemical
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some “progress report” should be given.
#'
#' @return A data.frame of document information and list presence tags
#' @export
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull list presence tags for BPA
#' bpa <- get_exposure_list_presence_tags(DTXSID = 'DTXSID7020182')

get_exposure_list_presence_tags_by_dtxsid <- function(DTXSID = NULL,
                                                      API_key = NULL,
                                                      Server = exposure_api_server,
                                                      verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/list-presence/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                          )
                        )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get general exposure prediction data
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @return A data.table of general exposure prediction data or NULL if data is
#' missing.
#' @export
#'
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull general exposure prediction data for BPA
#' bpa <- get_general_exposure_prediction(DTXSID = 'DTXSID7020182')

get_general_exposure_prediction <- function(DTXSID = NULL,
                                            API_key = NULL,
                                            Server = exposure_api_server,
                                            verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/seem/general/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    content <- httr::content(response, as = 'text', encoding = 'UTF-8')
    if (nchar(content) > 0){
      parsed_data <- jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8"))
      non_null_data <- null_to_na(parsed_data)
      return(data.table::as.data.table(non_null_data, rm.na = FALSE))
    }
    if (verbose) {
      print('The request was successful but returned no data')
    }
    return()
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get demographic exposure prediction data
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @return A data.table of demographic exposure prediction data.
#' @export
#'
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' # Pull general exposure prediction data for BPA
#' bpa <- get_demographic_exposure_prediction(DTXSID = 'DTXSID7020182')


get_demographic_exposure_prediction <- function(DTXSID = NULL,
                                                API_key = NULL,
                                                Server = exposure_api_server,
                                                verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input an DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/seem/demographic/search/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

null_to_na <- function(data_list){
  lapply(data_list, function(t){
    if (is.null(t)){
      return(NA_character_)
    }
    return(t)
  })
}

#' Get aggregate records by DTXSID
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame of aggregate record data by DTXSID.
#' @export
#'
#' @examplesIf FALSE
#' #Pull aggregate records for BPA by DTXSID
#' bpa_agg_records <- get_aggregate_records_by_dtxsid(DTXSID = 'DTXSID7020182')
get_aggregate_records_by_dtxsid <- function(DTXSID = NULL,
                                            API_key = NULL,
                                            Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                            verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/mmdb/aggregate/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get aggregate records by medium
#'
#' @param Medium The mmdb medium of exposure.
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param pageNumber Parameter for how to return data records.
#' @param verbose A logical indicating if some "progress report" should be
#'   given.
#'
#' @returns A list of search parameters and data of aggregate record data by
#'   medium.
#' @export
#'
#' @examplesIf FALSE
#' #Pull aggregate records for BPA by medium
#' bpa_agg_records <- get_aggregate_records_by_medium(Medium = 'surface water')
get_aggregate_records_by_medium <- function(Medium = NULL,
                                            API_key = NULL,
                                            Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                            pageNumber = 1,
                                            verbose = FALSE){
  if (is.null(Medium))
    stop('Please input a Medium!')

  if ((!is.integer(pageNumber) | pageNumber < 1) & verbose){
    warning('Setting `pageNumber` to 1!')
  }

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/mmdb/aggregate/by-medium?medium=', gsub(' ', '+', tolower(Medium)), ifelse(pageNumber>1, paste0('&pageNumber=',pageNumber), '')),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}




#' Get single sample records by DTXSID
#'
#' @param DTXSID The chemical identifier DTXSID
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be given.
#'
#' @returns A data.frame of single sample record data by DTXSID.
#' @export
#'
#' @examplesIf FALSE
#' #Pull single sample records for BPA by DTXSID
#' bpa_sample_records <- get_single_sample_records_by_dtxsid(DTXSID = 'DTXSID7020182')
get_single_sample_records_by_dtxsid <- function(DTXSID = NULL,
                                                API_key = NULL,
                                                Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                                verbose = FALSE){
  if (is.null(DTXSID))
    stop('Please input a DTXSID!')

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/mmdb/single-sample/by-dtxsid/', DTXSID),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Get single sample records by medium
#'
#' @param Medium The mmdb medium of exposure.
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param pageNumber Parameter for how to return data records.
#' @param verbose A logical indicating if some "progress report" should be
#'   given.
#'
#' @returns A list of search parameters and data of single sample record data by
#'   medium.
#' @export
#'
#' @examplesIf FALSE
#' #Pull single records for BPA by medium
#' bpa_sample_records <- get_single_sample_records_by_medium(Medium = 'surface water')

get_single_sample_records_by_medium <- function(Medium = NULL,
                                                API_key = NULL,
                                                Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                                pageNumber = 1,
                                                verbose = FALSE){
  if (is.null(Medium))
    stop('Please input a Medium!')

  if ((!is.integer(pageNumber) | pageNumber < 1) & verbose){
    warning('Setting `pageNumber` to 1!')
  }

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/mmdb/single-sample/by-medium?medium=', gsub(' ', '+', tolower(Medium)), ifelse(pageNumber>1, paste0('&pageNumber=',pageNumber), '')),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}


#' Retrieve MMDB medium categories
#'
#' @param API_key The user-specific API key
#' @param Server The root address for the API endpoint
#' @param verbose A logical indicating if some "progress report" should be
#'   given.
#'
#' @returns A data.frame of harmonized medium categories from MMDB and relevant
#'   descriptions.
#' @export
#'
#' @examplesIf FALSE
#' # Retrieve medium categories and descriptions
#' get_medium_categories()

get_medium_categories <- function(API_key = NULL,
                                  Server = 'https://comptox.epa.gov/ctx-api/exposure',
                                  verbose = FALSE){

  API_key <- check_api_key(API_key = API_key, verbose = verbose)
  if (is.null(API_key) & verbose){
    warning('Missing API key. Please supply during function call or save using `register_ctx_api_key()`!')
  }

  response <- httr::GET(url = paste0(Server, '/mmdb/mediums'),
                        httr::add_headers(.headers = c(
                          'Content-Type' =  'application/json',
                          'x-api-key' = API_key)
                        )
  )
  if(response$status_code == 401){
    stop(httr::content(response)$detail)
  }
  if(response$status_code == 200){
    return(jsonlite::fromJSON(httr::content(response, as = 'text', encoding = "UTF-8")))
  } else {
    if (verbose) {
      print(paste0('The request was unsuccessful, returning an error of ', response$status_code, '!'))
    }
  }
  return()
}

#' Exposure API Endpoint status
#'
#' @return Status of Exposure API Endpoints
#' @export
#'
#' @examplesIf has_ctx_key() & is.na(ctx_key() == 'FAKE_KEY')
#' status <- get_exposure_endpoint_status()
#' print(status)

get_exposure_endpoint_status <- function(){
  request <- httr::GET(url = paste0(exposure_api_server, "/health"))
  return(request$status_code)
}
