% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bal.compute.R
\name{bal.compute}
\alias{bal.compute}
\alias{bal.compute.bal.init}
\alias{bal.compute.default}
\alias{bal.init}
\alias{available.stats}
\title{Efficiently compute scalar balance statistics}
\usage{
bal.compute(x, ...)

\method{bal.compute}{bal.init}(x, weights = NULL, ...)

\method{bal.compute}{default}(x, treat = NULL, stat, s.weights = NULL, weights = NULL, ...)

bal.init(x, treat = NULL, stat, s.weights = NULL, ...)

available.stats(treat.type = "binary")
}
\arguments{
\item{x}{for \code{bal.compute()}, a \code{bal.init} object created by \code{bal.init()} or a matrix or data frame containing the covariates. For \code{bal.init()}, a matrix or data frame containing the covariates.}

\item{...}{other arguments used to specify options for the balance statistic. See Details for which arguments are allowed with each balance statistic. Ignored for the \code{bal.init} method of \code{bal.compute()}.}

\item{weights}{a vector of balancing weights to compute the weighted statistics.}

\item{treat}{a vector containing the treatment variable. Can be \code{NULL} to compute target balance statistics.}

\item{stat}{string; the name of the statistic to compute. See Details.}

\item{s.weights}{optional; a vector of sampling weights.}

\item{treat.type}{string; the treatment type, either \code{"binary"}, \code{"multinomial"}, \code{"continuous"}, or \code{"target"}. Abbreviations allowed.}
}
\value{
For \code{bal.compute()}, a single numeric value. For \code{bal.init()}, a \code{bal.init} object containing the components created in the initialization and the function used to compute the balance statistic. For \code{available.stats()}, a character vector of available statistics.
}
\description{
These are functions primarily designed for programmers who want to be able to quickly compute one of several scalar (single number) sample balance statistics, e.g., for use in selecting a tuning parameter when estimating balancing weights. \code{bal.compute()} computes a scalar balance statistics from the supplied inputs. \code{bal.init()} initializes the input so that when \code{bal.compute()} is used on the output along with a set of weights, the computation of the balance statistic is fast. \code{vignette("optimizing-balance")} provides an overview and more examples of how to use these functions. \code{available.stats()} returns the balance statistics available for the given treatment type.
}
\details{
The following list contains the allowable balance statistics that can be supplied to \code{bal.init()} or the default method of \code{bal.compute()}, the additional arguments that can be used with each one, and the treatment types allowed with each one. For all balance statistics, lower values indicate better balance. Target balance refers to the similarity between a sample and the same sample but weighted.
\describe{
\item{\code{smd.mean}, \code{smd.max}, \code{smd.rms}}{
The mean, maximum, or root-mean-squared absolute standardized mean difference, computed using \code{\link[=col_w_smd]{col_w_smd()}}. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, or \code{"ATC"}) to select the estimand (default is \code{"ATE"}), \code{focal} to identify the focal treatment group when the ATT is the estimand and the treatment has more than two categories, and \code{pairwise} to select whether mean differences should be computed between each pair of treatment groups or between each treatment group and the target group identified by \code{estimand} (default \code{TRUE}). Can be used with binary and multi-category treatments and for target balance.
}
\item{\code{ks.mean}, \code{ks.max}, \code{ks.rms}}{
The mean, maximum, or root-mean-squared Kolmogorov-Smirnov statistic, computed using \code{\link[=col_w_ks]{col_w_ks()}}. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, or \code{"ATC"}) to select the estimand (default is \code{"ATE"}), \code{focal} to identify the focal treatment group when the ATT is the estimand and the treatment has more than two categories, and \code{pairwise} to select whether statistics should be computed between each pair of treatment groups or between each treatment group and the target group identified by \code{estimand} (default \code{TRUE}). Can be used with binary and multi-category treatments and for target balance.
}
\item{\code{ovl.mean}, \code{ovl.max}, \code{ovl.rms}}{
The mean, maximum, or root-mean-squared overlapping coefficient complement, computed using \code{\link[=col_w_ovl]{col_w_ovl()}}. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, or \code{"ATC"}) to select the estimand (default is \code{"ATE"}), \code{integrate} to select whether integration is done using using \code{\link[=integrate]{integrate()}} (\code{TRUE}) or a Riemann sum (\code{FALSE}, the default), \code{focal} to identify the focal treatment group when the ATT is the estimand and the treatment has more than two categories, \code{pairwise} to select whether statistics should be computed between each pair of treatment groups or between each treatment group and the target group identified by \code{estimand} (default \code{TRUE}). Can be used with binary and multi-category treatments and for target balance.
}
\item{\code{mahalanobis}}{
The Mahalanobis distance between the treatment group means. This is similar to \code{smd.rms} but the covariates are standardized to remove correlations between them and de-emphasize redundant covariates. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, or \code{"ATC"}) to select the estimand (default is \code{"ATE"}) and \code{focal} to identify the focal treatment group when the ATT is the estimand. Can only be used with binary treatments and for target balance.
}
\item{\code{energy.dist}}{
The total energy distance between each treatment group and the target sample, which is a scalar measure of the similarity between two multivariate distributions. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, \code{"ATC"}, or \code{NULL}) to select the estimand (default is \code{NULL}), \code{focal} to identify the focal treatment group when the ATT is the estimand and the treatment has more than two categories, and \code{improved} to select whether the "improved" energy distance should be used when \code{estimand = "ATE"}, which emphasizes difference between treatment groups in addition to difference between each treatment group and the target sample (default \code{TRUE}). When \code{estimand = NULL}, only the energy distance between the treatment groups will be computed (i.e., as opposed to the energy distance between each treatment groups and the target sample). Can be used with binary and multi-category treatments and for target balance.
}
\item{\code{kernel.dist}}{
The kernel distance between the treatment groups, which is a scalar measure of the similarity between two multivariate distributions. Can only be used with binary treatments.
}
\item{\code{l1.med}}{
The median L1 statistic computed across a random selection of possible coarsening of the data. The other allowable arguments include \code{estimand} (\code{"ATE"}, \code{"ATT"}, or \code{"ATC"}) to select the estimand (default is \code{"ATE"}), \code{focal} to identify the focal treatment group when the ATT is the estimand and the treatment has more than two categories, \code{l1.min.bin} (default 2) and \code{l1.max.bin} default (12) to select the minimum and maximum number of bins with which to bin continuous variables and \code{l1.n} (default 101) to select the number of binnings used to select the binning at the median. \code{covs} should be supplied without splitting factors into dummies to ensure the binning works correctly; for simplicity, the \code{.covs} argument can be supplied, which will override \code{covs} but isn't used by other statistics. Can be used with binary and multi-category treatments.
}
\item{\code{r2}, \code{r2.2}, \code{r2.3}}{
The post-weighting \eqn{R^2} of a model for the treatment. The other allowable arguments include \code{poly} to add polynomial terms of the supplied order to the model and \code{int} (default \code{FALSE}) to add two-way interaction between covariates into the model. Using \code{r2.2} is a shortcut to requesting squares, and using \code{r2.3} is a shortcut to requesting cubes. Can be used with binary and continuous treatments. For binary treatments, the McKelvey and Zavoina \eqn{R^2} from a logistic regression is used; for continuous treatments, the \eqn{R^2} from a linear regression is used.
}
\item{\code{p.mean}, \code{p.max}, \code{p.rms}}{
The mean, maximum, or root-mean-squared absolute Pearson correlation between the treatment and covariates, computed using \code{\link[=col_w_corr]{col_w_corr()}}. Can only be used with continuous treatments.
}
\item{\code{s.mean}, \code{s.max}, \code{s.rms}}{
The mean, maximum, or root-mean-squared absolute Spearman correlation between the treatment and covariates, computed using \code{\link[=col_w_corr]{col_w_corr()}}. Can only be used with continuous treatments.
}
\item{\code{distance.cov}, \code{distance.cor}}{
The distance covariance or distance correlation, respectively, between the scaled covariates and treatment, which is a scalar measure of the independence of two possibly multivariate distributions. The distance correlation is scale-free and ranges from 0 (completely independent) to 1 (perfectly associated). Can only be used with continuous treatments.
}
}

Although statistics can be computed directly using \code{bal.compute()} alone, the intended workflow is to use \code{bal.init()} to initialize a \code{bal.init} object, which can then be passed to \code{bal.compute()} many times with different sets of weights, thereby minimizing the processing that \code{bal.init()} does because it is only done once. In contrast, using \code{bal.compute()} on covariates directly (i.e., using the default method) calls \code{bal.init()} internally each time, which can slow down evaluation. When speed isn't of interest or to calculate a balance statistic outside the context of balance optimization, the default method of \code{bal.compute()} can be a quick shortcut to avoid having to create a \code{bal.init} object first.
}
\examples{
\dontshow{if (requireNamespace("MatchIt", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Select the optimal number of subclasses for
# subclassification:
data("lalonde")
covs <- c("age", "educ", "race", "married",
          "nodegree", "re74", "re75")

# Estimate propensity score
p <- glm(reformulate(covs, "treat"),
         data = lalonde, 
         family = "binomial")$fitted.values

# Function to compute subclassification weights
subclass_ATE <- function(treat, p, nsub) {
    m <- MatchIt::matchit(treat ~ 1,
                          data = lalonde,
                          distance = p,
                          method = "subclass",
                          estimand = "ATE",
                          subclass = nsub)
    m$weights
}

# Initialize balance statistic; largest KS statistic
init <- bal.init(lalonde[covs],
                 treat = lalonde$treat, 
                 stat = "ks.max",
                 estimand = "ATE")

# Statistic prior to subclassification:
bal.compute(init)

# Testing 4 to 50 subclasses
nsubs <- 4:50
stats <- vapply(nsubs, function(n) {
    w <- subclass_ATE(lalonde$treat, p, n)
    bal.compute(init, w)
}, numeric(1L))

plot(stats ~ nsubs)

# 6 subclass gives lowest ks.max value (.238)
nsubs[which.min(stats)]
stats[which.min(stats)]

# See which statistics are available
available.stats("binary")
available.stats("multinomial")
available.stats("continuous")
available.stats("target")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link{balance-summary}}, \code{\link[=bal.tab]{bal.tab()}}

See \code{vignette("optimizing-balance")} for references and definitions of some of the above quantities.
}
