% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prop_ci_mn.R
\name{ci_prop_diff_mn_strata}
\alias{ci_prop_diff_mn_strata}
\title{Stratified Miettinen-Nurminen Confidence Interval for Difference in Proportions}
\usage{
ci_prop_diff_mn_strata(
  x,
  by,
  strata,
  method = c("score", "summary score"),
  conf.level = 0.95,
  delta = NULL,
  data = NULL
)
}
\arguments{
\item{x}{(\code{binary}/\code{numeric}/\code{logical})\cr
vector of a binary values, i.e. a logical vector, or numeric with values \code{c(0, 1)}}

\item{by}{(\code{string}) \cr A character or factor vector with exactly two unique levels
identifying the two groups to compare. Can also be a column name if a data
frame provided in the \code{data} argument.}

\item{strata}{(\code{numeric}) \cr A vector specifying the stratum for each observation. It needs
to be the length of x or a multiple of x if multiple levels of strata are
present. Can also be a column name (or vector of column names NOT quoted)
if a data frame provided in the \code{data} argument.}

\item{method}{(\code{string}) \cr Specifying how the CIs should be calculated. It must
equal either 'score' or 'summary score'. See details for more information
about the implementation differences.}

\item{conf.level}{(\verb{scalar numeric}) \cr a scalar in (0,1) indicating the confidence level. Default is 0.95}

\item{delta}{(\code{numeric}) \cr Optionally a single number or a vector of
numbers between -1 and 1 (not inclusive) to set the difference between two
groups under the null hypothesis. If provided, the function returns the
test statistic and p-value under the \code{delta}
hypothesis.}

\item{data}{(\code{data.frame}) \cr Optional data frame containing the variables specified in \code{x} and \code{by}.}
}
\value{
An object containing the following components:

\item{estimate}{The point estimate of the difference in proportions (p_x -
p_y)} \item{conf.low}{Lower bound of the confidence interval}
\item{conf.high}{Upper bound of the confidence interval}
\item{conf.level}{The confidence level used} \item{delta}{delta value(s) used}
\item{statistic}{Z-Statistic under the null hypothesis based on the given 'delta'}
\item{p.value}{p-value under the null hypothesis based on the given 'delta'}
\item{method}{Description of the method used ("Stratified \{method\}
Miettinen-Nurminen Confidence Interval")}

If \code{delta} is not provided statistic and p.value will be NULL
}
\description{
Calculates Stratified Miettinen-Nurminen (MN) confidence intervals and
corresponding point estimates for the difference between two proportions
}
\details{
The function implements the stratified Miettinen-Nurminen method to compute
confidence intervals for the difference between two proportions across multiple strata.
\deqn{H_0: \hat{d}-\delta <= 0 \qquad \text{vs.} \qquad H_1: \hat{d}-\delta > 0}

The "score" method is a weighted MN score first described in the original 1985
paper. The formula is:
\itemize{
\item Calculates weights for each stratum as \eqn{w_i = \frac{n_{xi} \cdot n_{yi}}{n_{xi} + n_{yi}}}
\item Computes the overall weighted difference \eqn{\hat{d} = \frac{\sum w_i \hat{p}_{xi}}{\sum w_i} -
          \frac{\sum w_i \hat{p}_{yi}}{\sum w_i}}
\item Uses the stratified test statistic: \deqn{Z_{\delta} = \frac{\hat{d} - \delta}
          {\sqrt{\sum_{i=1}^k \left(\frac{w_i}{\sum w_i}\right)^2 \cdot \hat{\sigma}_{mn}^2({d})}}}
\item Finds the range of all values of \eqn{\delta} for which the stratified test statistic (\eqn{Z_\delta})
falls in the acceptance region \eqn{\{ Z_\delta < z_{\alpha/2}\}}
}

The \eqn{\hat{\sigma}_{mn}^2(\hat{d})} is the Miettinen-Nurminen variance estimate.
See the details of \code{\link[=ci_prop_diff_mn]{ci_prop_diff_mn()}} for how \eqn{\hat{\sigma}_{mn}^2(\delta)} is calculated.

The "summary score" method follows the meta-analyses proposed in Agresti
2013 and is consistent with the "Summary Score Confidence Limits" method
used in SAS. The formula is:
\itemize{
\item The point estimate of the stratified risk difference is a weighted average of the midpoints of the within-stratum MN confidence intervals:
\deqn{
      \hat{d}_{\text{S}} = \sum_i \hat{d}_i w_i
      }
\item Define \eqn{s_i} as the width of the CI for the \eqn{i}th stratum divided by \eqn{2 \times z_{\alpha/2}} and then stratum weights are given by
\deqn{
    w_i = \left( \frac{1}{s_i^2} \right) \bigg/ \sum_i \left( \frac{1}{s_i^2} \right)
    }
\item The variance of \eqn{\hat{d}_{\text{S}} } is computed as
\deqn{
     \widehat{\text{Var}}(\hat{d}_{\text{S}}) = \frac{1}{\sum_i \left( \frac{1}{s_i^2} \right) }
    }
\item Confidence limits for the stratified risk difference estimate are
\deqn{
    \hat{d}_{\text{S}} \pm \left( z_{\alpha /2} \times \widehat{\text{Var}}(\hat{d}_{\text{S}}) \right)
    }
}
}
\examples{
# Generate binary samples with strata
responses <- expand(c(9, 3, 7, 2), c(10, 10, 10, 10))
arm <- rep(c("treat", "control"), 20)
strata <- rep(c("stratum1", "stratum2"), times = c(20, 20))

# Calculate stratified confidence interval for difference in proportions
ci_prop_diff_mn_strata(x = responses, by = arm, strata = strata)

# Using the summary score method
ci_prop_diff_mn_strata(x = responses, by = arm, strata = strata,
                      method = "summary score")

# Calculate 99\% confidence interval
ci_prop_diff_mn_strata(x = responses, by = arm, strata = strata,
                      conf.level = 0.99)

# Calculate p-value under null hypothesis delta = 0.2
ci_prop_diff_mn_strata(x = responses, by = arm, strata = strata,
                      delta = 0.2)

}
\references{
Miettinen, O. S., & Nurminen, M. (1985). Comparative analysis of two rates.
Statistics in Medicine, 4(2), 213-226.

\href{https://documentation.sas.com/doc/en/pgmsascdc/9.4_3.5/procstat/procstat_freq_details63.htm}{Common Risk Difference :: Base SAS(R) 9.4 Procedures Guide: Statistical Procedures, Third Edition}

Agresti, A. (2013). Categorical Data Analysis. 3rd Edition. John Wiley & Sons, Hoboken, NJ
}
