#' Parametric Bootstrap Likelihood Ratio Test (PB_LRT)
#'
#' Performs a parametric bootstrap likelihood ratio test for comparing group means
#' under heteroscedasticity (unequal variances). This test serves as a robust alternative to classical one-way ANOVA
#' when the assumption of equal variances is violated.
#'
#' @param means A numeric vector of group sample means.
#' @param vars A numeric vector of group sample variances.
#' @param ns A numeric vector of group sample sizes.
#' @param tol Convergence tolerance for iterative re-estimation under the null hypothesis. Default is 1e-5.
#' @param H Number of bootstrap iterations. Default is 1000.
#' @param alpha Significance level for the hypothesis test. Default is 0.05.
#'
#' @return A list of class \code{"PB_LRT"} containing:
#' \item{test_statistic}{Observed value of the likelihood ratio statistic.}
#' \item{critical_value}{Bootstrap-based critical value under the null hypothesis.}
#' \item{decision}{Conclusion of the hypothesis test.}
#'
#' @examples
#' # Example with 3 groups
#' set.seed(123)
#' means <- c(5.1, 6.3, 7.0)
#' vars <- c(1.2, 1.8, 2.5)
#' ns <- c(20, 25, 22)
#' result <- PB_LRT(means, vars, ns)
#' print(result)
#'
#' @importFrom stats rnorm var quantile
#' @export
PB_LRT <- function(means, vars, ns, tol = 1e-5, H = 1000, alpha = 0.05) {
  k <- length(means)
  n <- sum(ns)

  mu0 <- sum(ns * means) / n
  s0 <- vars

  repeat {
    u <- ns / s0
    mun <- sum(u * means) / sum(u)

    sig_n <- ((ns - 1) * vars + ns * (means - mun)^2) / ns

    if (abs(mu0 - mun) < tol) break

    mu0 <- mun
    s0 <- sig_n
  }

  num <- prod(vars^(ns / 2))
  den <- prod(sig_n^(ns / 2))
  lambda_obs <- num / den

  lambda_star <- numeric(H)

  for (h in 1:H) {
    x_boot <- lapply(1:k, function(i) rnorm(ns[i], mean = 0, sd = sqrt(vars[i])))

    mu_hat_boot <- vapply(x_boot, mean, numeric(1))
    sigmasq_hat_boot <- vapply(1:k, function(i) mean((x_boot[[i]] - mu_hat_boot[i])^2), numeric(1))

    mu_old_b <- mean(mu_hat_boot)
    sigma_vec_old_b <- sigmasq_hat_boot

    repeat {
      weights_b <- ns / sigma_vec_old_b
      mu_new_b <- sum(weights_b * mu_hat_boot) / sum(weights_b)
      if (abs(mu_new_b - mu_old_b) < tol) break
      sigma_vec_old_b <- vapply(1:k, function(i) mean((x_boot[[i]] - mu_new_b)^2), numeric(1))
      mu_old_b <- mu_new_b
    }

    sigmasq_H0_b <- sigma_vec_old_b
    lambda_star[h] <- prod((sigmasq_hat_boot / sigmasq_H0_b)^(ns / 2))
  }
  lambda_star_sorted <- sort(lambda_star)
  critical_index <- ceiling((alpha) * H)
  critical_value <- lambda_star_sorted[critical_index]

  decision <- if (lambda_obs<critical_value) {
    "Reject H0: At least one pair of means differs significantly."
  } else {
    "Do not reject H0: No significant difference among group means."
  }

  result <- list(
    test_statistic = lambda_obs,
    critical_value = critical_value,
    decision = decision
  )
  class(result) <- "PB_LRT"
  return(result)
}



