% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bgm.R
\name{bgm}
\alias{bgm}
\title{Bayesian Estimation or Edge Selection for Markov Random Fields}
\usage{
bgm(
  x,
  variable_type = "ordinal",
  baseline_category,
  iter = 1000,
  warmup = 1000,
  pairwise_scale = 2.5,
  main_alpha = 0.5,
  main_beta = 0.5,
  edge_selection = TRUE,
  edge_prior = c("Bernoulli", "Beta-Bernoulli", "Stochastic-Block"),
  inclusion_probability = 0.5,
  beta_bernoulli_alpha = 1,
  beta_bernoulli_beta = 1,
  dirichlet_alpha = 1,
  lambda = 1,
  na_action = c("listwise", "impute"),
  update_method = c("nuts", "adaptive-metropolis", "hamiltonian-mc"),
  target_accept,
  hmc_num_leapfrogs = 100,
  nuts_max_depth = 10,
  learn_mass_matrix = FALSE,
  chains = 4,
  cores = parallel::detectCores(),
  display_progress = c("per-chain", "total", "none"),
  seed = NULL,
  interaction_scale,
  burnin,
  save,
  threshold_alpha,
  threshold_beta
)
}
\arguments{
\item{x}{A data frame or matrix with \code{n} rows and \code{p} columns
containing binary and ordinal responses. Variables are automatically
recoded to non-negative integers (\code{0, 1, ..., m}). For regular
ordinal variables, unobserved categories are collapsed; for
Blume–Capel variables, all categories are retained.}

\item{variable_type}{Character or character vector. Specifies the type of
each variable in \code{x}. Allowed values: \code{"ordinal"} or
\code{"blume-capel"}. Binary variables are automatically treated as
\code{"ordinal"}. Default: \code{"ordinal"}.}

\item{baseline_category}{Integer or vector. Baseline category used in
Blume–Capel variables. Can be a single integer (applied to all) or a
vector of length \code{p}. Required if at least one variable is of type
\code{"blume-capel"}.}

\item{iter}{Integer. Number of post–burn-in iterations (per chain).
Default: \code{1e3}.}

\item{warmup}{Integer. Number of warmup iterations before collecting
samples. A minimum of 1000 iterations is enforced, with a warning if a
smaller value is requested. Default: \code{1e3}.}

\item{pairwise_scale}{Double. Scale of the Cauchy prior for pairwise
interaction parameters. Default: \code{2.5}.}

\item{main_alpha, main_beta}{Double. Shape parameters of the
beta-prime prior for threshold parameters. Must be positive. If equal,
the prior is symmetric. Defaults: \code{main_alpha = 0.5} and
\code{main_beta = 0.5}.}

\item{edge_selection}{Logical. Whether to perform Bayesian edge selection.
If \code{FALSE}, the model estimates all edges. Default: \code{TRUE}.}

\item{edge_prior}{Character. Specifies the prior for edge inclusion.
Options: \code{"Bernoulli"}, \code{"Beta-Bernoulli"}, or
\code{"Stochastic-Block"}. Default: \code{"Bernoulli"}.}

\item{inclusion_probability}{Numeric scalar. Prior inclusion probability
of each edge (used with the Bernoulli prior). Default: \code{0.5}.}

\item{beta_bernoulli_alpha, beta_bernoulli_beta}{Double. Shape parameters
for the beta distribution in the Beta–Bernoulli and the Stochastic-Block
priors. Must be positive. Defaults: \code{beta_bernoulli_alpha = 1} and
\code{beta_bernoulli_beta = 1}.}

\item{dirichlet_alpha}{Double. Concentration parameter of the Dirichlet
prior on block assignments (used with the Stochastic Block model).
Default: \code{1}.}

\item{lambda}{Double. Rate of the zero-truncated Poisson prior on the
number of clusters in the Stochastic Block Model. Default: \code{1}.}

\item{na_action}{Character. Specifies missing data handling. Either
\code{"listwise"} (drop rows with missing values) or \code{"impute"}
(perform single imputation during sampling). Default: \code{"listwise"}.}

\item{update_method}{Character. Specifies how the MCMC sampler updates
the model parameters:
\describe{
  \item{"adaptive-metropolis"}{Componentwise adaptive Metropolis–Hastings
    with Robbins–Monro proposal adaptation.}
  \item{"hamiltonian-mc"}{Hamiltonian Monte Carlo with fixed path length
    (number of leapfrog steps set by \code{hmc_num_leapfrogs}).}
  \item{"nuts"}{The No-U-Turn Sampler, an adaptive form of HMC with
    dynamically chosen trajectory lengths.}
}
Default: \code{"nuts"}.}

\item{target_accept}{Numeric between 0 and 1. Target acceptance rate for
the sampler. Defaults are set automatically if not supplied:
\code{0.44} for adaptive Metropolis, \code{0.65} for HMC,
and \code{0.60} for NUTS.}

\item{hmc_num_leapfrogs}{Integer. Number of leapfrog steps for Hamiltonian
Monte Carlo. Must be positive. Default: \code{100}.}

\item{nuts_max_depth}{Integer. Maximum tree depth in NUTS. Must be positive.
Default: \code{10}.}

\item{learn_mass_matrix}{Logical. If \code{TRUE}, adapt a diagonal mass
matrix during warmup (HMC/NUTS only). If \code{FALSE}, use the identity
matrix. Default: \code{FALSE}.}

\item{chains}{Integer. Number of parallel chains to run. Default: \code{4}.}

\item{cores}{Integer. Number of CPU cores for parallel execution.
Default: \code{parallel::detectCores()}.}

\item{display_progress}{Logical. Whether to show a progress bar during
sampling. Default: \code{TRUE}.}

\item{seed}{Optional integer. Random seed for reproducibility. Must be a
single non-negative integer.}

\item{interaction_scale, burnin, save, threshold_alpha, threshold_beta}{`r lifecycle::badge("deprecated")`
Deprecated arguments as of **bgms 0.1.6.0**.
Use `pairwise_scale`, `warmup`, `main_alpha`, and `main_beta` instead.}
}
\value{
A list of class \code{"bgms"} with posterior summaries, posterior mean
matrices, and access to raw MCMC draws. The object can be passed to
\code{print()}, \code{summary()}, and \code{coef()}.

Main components include:
\itemize{
  \item \code{posterior_summary_main}: Data frame with posterior summaries
    (mean, sd, MCSE, ESS, Rhat) for category threshold parameters.
  \item \code{posterior_summary_pairwise}: Data frame with posterior
    summaries for pairwise interaction parameters.
  \item \code{posterior_summary_indicator}: Data frame with posterior
    summaries for edge inclusion indicators (if \code{edge_selection = TRUE}).

  \item \code{posterior_mean_main}: Matrix of posterior mean thresholds
    (rows = variables, cols = categories or parameters).
  \item \code{posterior_mean_pairwise}: Symmetric matrix of posterior mean
    pairwise interaction strengths.
  \item \code{posterior_mean_indicator}: Symmetric matrix of posterior mean
    inclusion probabilities (if edge selection was enabled).

  \item  Additional summaries returned when
    \code{edge_prior = "Stochastic-Block"}. For more details about this prior
    see \insertCite{SekulovskiEtAl_2025;textual}{bgms}.
   \itemize{
      \item \code{posterior_summary_pairwise_allocations}: Data frame with
      posterior summaries (mean, sd, MCSE, ESS, Rhat) for the pairwise
      cluster co-occurrence of the nodes. This serves to indicate
      whether the estimated posterior allocations,co-clustering matrix
      and posterior cluster probabilities (see blow) have converged.
      \item \code{posterior_coclustering_matrix}: a symmetric matrix of
      pairwise proportions of occurrence of every variable. This matrix
      can be plotted to visually inspect the estimated number of clusters
      and visually inspect nodes that tend to switch clusters.
      \item \code{posterior_mean_allocations}: A vector with the posterior mean
      of the cluster allocations of the nodes. This is calculated using the method
      proposed in \insertCite{Dahl2009;textual}{bgms}.
      \item \code{posterior_mode_allocations}: A vector with the posterior
       mode of the cluster allocations of the nodes.
      \item \code{posterior_num_blocks}: A data frame with the estimated
      posterior inclusion probabilities for all the possible number of clusters.
      }
  \item \code{raw_samples}: A list of raw MCMC draws per chain:
    \describe{
      \item{\code{main}}{List of main effect samples.}
      \item{\code{pairwise}}{List of pairwise effect samples.}
      \item{\code{indicator}}{List of indicator samples
        (if edge selection enabled).}
      \item{\code{allocations}}{List of cluster allocations
        (if SBM prior used).}
      \item{\code{nchains}}{Number of chains.}
      \item{\code{niter}}{Number of post–warmup iterations per chain.}
      \item{\code{parameter_names}}{Named lists of parameter labels.}
    }

  \item \code{arguments}: A list of function call arguments and metadata
    (e.g., number of variables, warmup, sampler settings, package version).
}

The \code{summary()} method prints formatted posterior summaries, and
\code{coef()} extracts posterior mean matrices.

NUTS diagnostics (tree depth, divergences, energy, E-BFMI) are included
in \code{fit$nuts_diag} if \code{update_method = "nuts"}.
}
\description{
The \code{bgm} function estimates the pseudoposterior distribution of
category thresholds (main effects) and pairwise interaction parameters of a
Markov Random Field (MRF) model for binary and/or ordinal variables.
Optionally, it performs Bayesian edge selection using spike-and-slab
priors to infer the network structure.
}
\details{
This function models the joint distribution of binary and ordinal variables
using a Markov Random Field, with support for edge selection through Bayesian
variable selection. The statistical foundation of the model is described in
\insertCite{MarsmanVandenBerghHaslbeck_2024;textual}{bgms}, where the ordinal
MRF model and its Bayesian estimation procedure were first introduced. While
the implementation in \pkg{bgms} has since been extended and updated (e.g.,
alternative priors, parallel chains, HMC/NUTS warmup), it builds on that
original framework.

Key components of the model are described in the sections below.
}
\section{Ordinal Variables}{

The function supports two types of ordinal variables:

\strong{Regular ordinal variables}:
Assigns a category threshold parameter to each response category except the
lowest. The model imposes no additional constraints on the distribution of
category responses.

\strong{Blume-Capel ordinal variables}:
Assume a baseline category (e.g., a “neutral” response) and score responses
by distance from this baseline. Category thresholds are modeled as:

\deqn{\mu_{c} = \alpha \cdot c + \beta \cdot (c - b)^2}

where:
\itemize{
  \item \eqn{\mu_{c}}: category threshold for category \eqn{c}
  \item \eqn{\alpha}: linear trend across categories
  \item \eqn{\beta}: preference toward or away from the baseline
   \itemize{
     \item If \eqn{\beta < 0}, the model favors responses near the baseline
     category;
     \item if \eqn{\beta > 0}, it favors responses farther away (i.e.,
     extremes).
   }
  \item \eqn{b}: baseline category
}
}

\section{Edge Selection}{

When \code{edge_selection = TRUE}, the function performs Bayesian variable
selection on the pairwise interactions (edges) in the MRF using
spike-and-slab priors.

Supported priors for edge inclusion:
\itemize{
  \item \strong{Bernoulli}: Fixed inclusion probability across edges.
  \item \strong{Beta-Bernoulli}: Inclusion probability is assigned a Beta
  prior distribution.
  \item \strong{Stochastic-Block}: Cluster-based edge priors with Beta,
  Dirichlet, and Poisson hyperpriors.
}

All priors operate via binary indicator variables controlling the inclusion
or exclusion of each edge in the MRF.
}

\section{Prior Distributions}{


\itemize{
  \item \strong{Pairwise effects}: Modeled with a Cauchy (slab) prior.
  \item \strong{Main effects}: Modeled using a beta-prime
  distribution.
  \item \strong{Edge indicators}: Use either a Bernoulli, Beta-Bernoulli, or
  Stochastic-Block prior (as above).
}
}

\section{Sampling Algorithms and Warmup}{


Parameters are updated within a Gibbs framework, but the conditional
updates can be carried out using different algorithms:
\itemize{
  \item \strong{Adaptive Metropolis–Hastings}: Componentwise random–walk
    updates for main effects and pairwise effects. Proposal standard
    deviations are adapted during burn–in via Robbins–Monro updates
    toward a target acceptance rate.

  \item \strong{Hamiltonian Monte Carlo (HMC)}: Joint updates of all
    parameters using fixed–length leapfrog trajectories. Step size is
    tuned during warmup via dual–averaging; the diagonal mass matrix can
    also be adapted if \code{learn_mass_matrix = TRUE}.

  \item \strong{No–U–Turn Sampler (NUTS)}: An adaptive extension of HMC
    that dynamically chooses trajectory lengths. Warmup uses a staged
    adaptation schedule (fast–slow–fast) to stabilize step size and, if
    enabled, the mass matrix.
}

When \code{edge_selection = TRUE}, updates of edge–inclusion indicators
are carried out with Metropolis–Hastings steps. These are switched on
after the core warmup phase, ensuring that graph updates occur only once
the samplers’ tuning parameters (step size, mass matrix, proposal SDs)
have stabilized.

After warmup, adaptation is disabled. Step size and mass matrix are
fixed at their learned values, and proposal SDs remain constant.
}

\section{Warmup and Adaptation}{


The warmup procedure in \code{bgm} is based on the multi–stage adaptation
schedule used in Stan \insertCite{stan-manual}{bgms}. Warmup iterations are
split into several phases:

\itemize{
  \item \strong{Stage 1 (fast adaptation)}: A short initial interval
    where only step size (for HMC/NUTS) is adapted, allowing the chain
    to move quickly toward the typical set.

  \item \strong{Stage 2 (slow windows)}: A sequence of expanding,
    memoryless windows where both step size and, if
    \code{learn_mass_matrix = TRUE}, the diagonal mass matrix are
    adapted. Each window ends with a reset of the dual–averaging scheme
    for improved stability.

  \item \strong{Stage 3a (final fast interval)}: A short interval at the
    end of the core warmup where the step size is adapted one final time.

  \item \strong{Stage 3b (proposal–SD tuning)}: Only active when
    \code{edge_selection = TRUE} under HMC/NUTS. In this phase,
    Robbins–Monro adaptation of proposal standard deviations is
    performed for the Metropolis steps used in edge–selection moves.

  \item \strong{Stage 3c (graph selection warmup)}: Also only relevant
    when \code{edge_selection = TRUE}. At the start of this phase, a
    random graph structure is initialized, and Metropolis–Hastings
    updates for edge inclusion indicators are switched on.
}

When \code{edge_selection = FALSE}, the total number of warmup iterations
equals the user–specified \code{burnin}. When \code{edge_selection = TRUE}
and \code{update_method} is \code{"nuts"} or \code{"hamiltonian-mc"},
the schedule automatically appends additional Stage-3b and Stage-3c
intervals, so the total warmup is strictly greater than the requested
\code{burnin}.

After all warmup phases, the sampler transitions to the sampling phase
with adaptation disabled. Step size and mass matrix (for HMC/NUTS) are
fixed at their learned values, and proposal SDs remain constant.

This staged design improves stability of proposals and ensures that both
local parameters (step size) and global parameters (mass matrix, proposal
SDs) are tuned before collecting posterior samples.

For adaptive Metropolis–Hastings runs, step size and mass matrix
adaptation are not relevant. Proposal SDs are tuned continuously during
burn–in using Robbins–Monro updates, without staged fast/slow intervals.
}

\section{Missing Data}{


If \code{na_action = "listwise"}, observations with missing values are
removed.
If \code{na_action = "impute"}, missing values are imputed during Gibbs
sampling.
}

\examples{
\donttest{
# Run bgm on subset of the Wenchuan dataset
fit = bgm(x = Wenchuan[, 1:5])

# Posterior inclusion probabilities
summary(fit)$indicator

# Posterior pairwise effects
summary(fit)$pairwise
}

}
\references{
\insertAllCited{}
}
\seealso{
\code{vignette("intro", package = "bgms")} for a worked example.
}
