% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aqp-label-placement-solvers.R
\name{electroStatics_1D}
\alias{electroStatics_1D}
\title{Label placement based on a simulation of electrostatic forces}
\usage{
electroStatics_1D(
  x,
  thresh,
  q = 1,
  chargeDecayRate = 0.01,
  QkA_GrowthRate = 0.05,
  maxIter = 100,
  tiny = 1e-04,
  const = 0.001,
  trace = FALSE,
  ...
)
}
\arguments{
\item{x}{numeric vector, pre-sorted sorted, without duplication, describing 1D label (particle) configuration}

\item{thresh}{numeric, overlap threshold, same as in \code{\link[=fixOverlap]{fixOverlap()}}}

\item{q}{numeric, electrical charge (typically between 0.1 and 2)}

\item{chargeDecayRate}{numeric, exponential decay rate constant for \code{q} as a function of iteration \code{i}}

\item{QkA_GrowthRate}{numeric, growth rate constant for \code{Qk} applied to attraction to uniform spacing of labels, invoked when rank order is violated during the simulation}

\item{maxIter}{integer, maximum number of iterations before giving up}

\item{tiny}{numeric, 0-values replaced by this number to avoid division by 0 and infinite forces}

\item{const}{numeric, empirical constant added to the 1D electrostatic force equation to dampen oscillation: \code{(Qk * Q1 * Q2) / (d^ex + const)}}

\item{trace}{logical, include diagnostic output}

\item{\dots}{not used, absorbs additional arguments to \code{\link[=fixOverlap]{fixOverlap()}}}
}
\value{
When \code{trace = TRUE} a \code{list}, otherwise numeric vector with \code{converged} attribute.
}
\description{
This function attempts to move labels along a 1D coordinate system such that overlap (as specified by threshold) is minimized. An electrostatic simulation applies forces of repulsion between labels that are within \code{thresh} (e.g. overlapping) and forces of attraction to a uniformly spaced sequence to iteratively perturb affected labels until either no overlap is reported, or a maximum number of iterations (\code{maxIter}) has been reached.
}
\details{
Difficult overlap problems can be addressed by reducing \code{thresh} and increasing \code{q}. Large values of \code{q} can lead to chaotic results.

This function will generate unpredictable output when \code{x} contains duplicate values.

This function requires input to be pre-sorted, although interesting "artistic" simulations will often result from unsorted \code{x}.
}
\examples{

# vector of object locations, with potential overlap
x <- c(1, 2, 3, 3.3, 3.8, 5, 6, 7, 8, 9, 10)

# full diagnostic output
z <- electroStatics_1D(x, thresh = 0.65, trace = TRUE, q = 1)
txt <- sprintf("Converged \%s (\%s iterations)", z$converged, length(z$cost))

plot(
seq_along(z$cost),
z$cost, 
las = 1, 
xlab = 'Iteration', 
ylab = 'Overlap Cost', 
type = 'b', 
main = txt
)

abline(h = 0, lty = 2, col = 2)

# final configuration only
xnew <- electroStatics_1D(x, thresh = 0.65, q = 1)

# check for convergence
attr(xnew, 'converged')

# compare original vs. modified
data.frame(orig = x, new = round(xnew, 2))

}
\seealso{
\code{\link[=fixOverlap]{fixOverlap()}}, \code{\link[=SANN_1D]{SANN_1D()}}
}
\author{
D.E. Beaudette and K.C. Thompson
}
