% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/walsNB.R
\name{walsNBfit}
\alias{walsNBfit}
\title{Fitter function for Weighted Average Least Squares estimation of NB2 regression model}
\usage{
walsNBfit(
  X1,
  X2,
  y,
  betaStart1,
  betaStart2,
  rhoStart,
  family,
  prior,
  method = c("fullSVD", "original"),
  svdTol = .Machine$double.eps,
  svdRtol = 1e-06,
  keepUn = FALSE,
  keepR = FALSE,
  eigenSVD = TRUE,
  postmult = TRUE,
  ...
)
}
\arguments{
\item{X1}{Design matrix for focus regressors. Usually includes a constant
(column full of 1s) and can be generated using \code{\link[stats]{model.matrix}}.}

\item{X2}{Design matrix for auxiliary regressors. Usually does not include
a constant column and can also be generated using \code{\link[stats]{model.matrix}}.}

\item{y}{Count response as vector.}

\item{betaStart1}{Starting values for coefficients of focus regressors X1.}

\item{betaStart2}{Starting values for coefficients of auxiliary regressors X2.}

\item{rhoStart}{Starting value for log-dispersion parameter of NB2}

\item{family}{Object of class \code{"\link[WALS]{familyNBWALS}"}. Currently only supports
\code{\link[WALS]{negbinWALS}}.}

\item{prior}{Object of class \code{"\link[WALS]{familyPrior}"}. For example
\code{\link[WALS]{weibull}} or \code{\link[WALS]{laplace}}.}

\item{method}{Specifies method used. Available methods are \code{"fullSVD"}
(default) or \code{"original"}. See details.}

\item{svdTol}{Tolerance for rank of matrix \eqn{\bar{Z}_{1}} and \eqn{\bar{Z}}.
Only used if \code{method = "fullSVD"}.
Checks if smallest eigenvalue in SVD of \eqn{\bar{Z}_1} and \eqn{\bar{Z}}
is larger than \code{svdTol}, otherwise reports a rank deficiency.}

\item{svdRtol}{Relative tolerance for rank of matrix \eqn{\bar{Z}_{1}} and \eqn{\bar{Z}}.
Only used if \code{method = "fullSVD"}. Checks if ratio of largest to smallest
eigenvalue in SVD of \eqn{\bar{Z}_1} and \eqn{\bar{Z}} is larger than
\code{svdRtol}, otherwise reports a rank deficiency.}

\item{keepUn}{If \code{TRUE}, keeps the one-step ML estimators of the
unrestricted model, i.e. \eqn{\tilde{\gamma}_{u}} and \eqn{\tilde{\beta}_{u}}.}

\item{keepR}{If \code{TRUE}, keeps the one-step ML estimators of the fully
restricted model, i.e. \eqn{\tilde{\gamma}_{r}} and \eqn{\tilde{\beta}_{r}}.}

\item{eigenSVD}{If \code{TRUE}, then \code{semiorthogonalize()} uses \code{svd()}
to compute the eigendecomposition of \eqn{\bar{\Xi}} instead of \code{eigen()}.
In this case, the tolerances of \code{svdTol} and \code{svdRtol} are used to
determine whether \eqn{\bar{\Xi}} is of full rank (need it for \eqn{\bar{\Xi}^{-1/2}}).}

\item{postmult}{If \code{TRUE} (default), then it computes
\deqn{\bar{Z}_{2} = \bar{X}_{2} \bar{\Delta}_{2} \bar{T} \bar{\Lambda}^{-1/2} \bar{T}^{\top},}
where \eqn{\bar{T}} contains the eigenvectors and \eqn{\bar{\Lambda}} the
eigenvalues from the eigenvalue decomposition
\deqn{\bar{\Xi} = \bar{T} \bar{\Lambda} \bar{T}^{\top},}
instead of
\deqn{\bar{Z}_{2} = \bar{X}_{2} \bar{\Delta}_{2} \bar{T} \bar{\Lambda}^{-1/2}.}
See \insertCite{huynhwals;textual}{WALS} for more details. The latter is used
in the original MATLAB code for WALS in the linear regression model,
see eq. (12) of \insertCite{magnus2016wals;textual}{WALS}.
The first form is required in eq. (9) of \insertCite{deluca2018glm;textual}{WALS}.
\strong{Thus, it is not recommended to set \code{postmult = FALSE}.}}

\item{...}{Arguments for internal function \code{\link[WALS]{computePosterior}}.}
}
\value{
A list containing
\item{coef}{Model averaged estimates of all coefficients.}
\item{beta1}{Model averaged estimates of the coefficients of the focus regressors.}
\item{beta2}{Model averaged estimates of the coefficients of the auxiliary regressors.}
\item{rho}{Model averaged estimate of the log-dispersion parameter of the
NB2 distribution.}
\item{gamma1}{Model averaged estimates of the coefficients of the transformed
focus regressors.}
\item{gamma2}{Model averaged estimates of the coefficients of the transformed
auxiliary regressors.}
\item{condition}{Condition number of the matrix
\eqn{\bar{\Xi} = \bar{\Delta}_{2} \bar{X}_{2}^{\top} \bar{M}_{1} \bar{X}_{2} \bar{\Delta}_{2}}.}
\item{vcovBeta}{\code{NULL}, not implemented yet, placeholder for estimated
covariance matrix of the regression coefficients.}
\item{vcovGamma}{\code{NULL}, not implemented yet, placeholder for estimated
covariance matrix of the coefficients of the transformed regressors.}
\item{betaStart}{Starting values of the regression coefficients for the
one-step ML estimators.}
\item{rhoStart}{Starting values of the dispersion parameter for the
one-step ML estimators.}
\item{method}{Stores \code{method} used from the arguments.}
\item{prior}{\code{familyPrior}. The \code{prior} specified in the arguments.}
\item{betaUn1}{If \code{keepUn = TRUE}, contains the unrestricted one-step ML
estimators of the coefficients of the focus regressors. Else \code{NULL}.}
\item{betaUn2}{If \code{keepUn = TRUE}, contains the unrestricted one-step ML
estimators of the coefficients of the auxiliary regressors. Else \code{NULL}.}
\item{gammaUn1}{If \code{keepUn = TRUE}, contains the unrestricted one-step ML
estimators of the coefficients of the transformed focus regressors. Else \code{NULL}.}
\item{gammaUn2}{If \code{keepUn = TRUE}, contains the unrestricted one-step ML
estimators of the coefficients of the transformed auxiliary regressors. Else \code{NULL}.}
\item{gamma1r}{If \code{keepR = TRUE}, contains the fully restricted one-step
ML estimator for the transformed regressors (only focus regressors).
Else \code{NULL}.}
\item{k1}{Number of focus regressors.}
\item{k2}{Number of auxiliary regressors.}
\item{n}{Number of observations.}
\item{X1names}{Names of the focus regressors.}
\item{X2names}{Names of the auxiliary regressors.}
\item{familyStart}{The family object of class \code{"\link{familyNBWALS}"} used for the
estimation of the starting values.}
\item{family}{The family object of class \code{"\link{familyNBWALS}"} used later for predictions.}
\item{fitted.link}{Linear link fitted to the data.}
\item{fitted.values}{Estimated conditional mean for the data. Lives on the
scale of the response.}
}
\description{
Workhorse function behind \code{\link{walsNB}} and used internally in
\code{\link{walsNBfitIterate}}.
}
\details{
The method to be specified in \code{method} mainly differ in the way
they compute the fully restricted and unrestricted estimators for the
transformed regressors \eqn{Z}, i.e. \eqn{\tilde{\gamma}_{1r}},
and \eqn{\tilde{\gamma}_{u}}.

\describe{
\item{"fullSVD"}{Recommended approach. First applies an SVD to \eqn{\bar{Z}_{1}}
to compute \eqn{\bar{X}_{2}^{\top} \bar{M}_{1} \bar{X}_{2}}:
It is used for computing the inverse of

\deqn{\bar{X}_{1}^{\top}\bar{X}_{1}
 + \bar{g} \bar{\epsilon} X_{1}^{\top}\bar{q} \bar{q}^{\top} X_{1},}

when using the Sherman-Morrison-Woodbury formula. We further leverage the
SVD of \eqn{\bar{Z}_1} and additionally \eqn{\bar{Z}} to compute the
unrestricted estimator \eqn{\tilde{\gamma}_{u}} and the fully restricted
estimator \eqn{\tilde{\gamma}_{r}}. For \eqn{\tilde{\gamma}_{u}}, we simply
use the SVD of \eqn{\bar{Z}} to solve the full equation system derived from
the one-step ML problem for more details. The SVD of \eqn{\bar{Z}_1} is further
used in computing the model averaged estimator for the focus regressors
\eqn{\hat{\gamma}_1}.

Described in more detail in the appendix of \insertCite{huynhwals;textual}{WALS}.}

\item{"original"}{Computes all inverses directly using \code{\link[base]{solve}}
and does not make use of the Sherman-Morrison-Woodbury formula for certain
inverses. Specifically, it directly inverts the matrix
\eqn{\bar{Z}_{1}^{\top} \bar{Z}_{1}} using \code{\link[base]{solve}}
in order to compute \eqn{\bar{M}_1}. Moreover, it computes the fully
unrestricted estimators of the focus regressors
\eqn{\tilde{\gamma}_{1u}} and of the auxiliary regressors
\eqn{\tilde{\gamma}_{2u}} and the fully restricted estimator
\eqn{\tilde{\gamma}_{1r}} by directly implementing the formulas derived
in \insertCite{huynhwalsnb;textual}{WALS}.
This method should only be used as reference and for easier
debugging.}
}

All variables in the code that contain "start" in their name are computed
using the starting values of the one-step ML estimators. See section
"One-step ML estimator" of \insertCite{huynhwalsnb}{WALS} for details.
}
\examples{
data("NMES1988", package = "AER")
NMES1988 <- na.omit(NMES1988)
form <- (visits ~ health + chronic + age + insurance + adl + region + gender
         + married + income + school + employed)
X <- model.matrix(form, data = NMES1988)
focus <- c("(Intercept)", "healthpoor", "healthexcellent", "chronic", "age",
        "insuranceyes")
aux <- c("adllimited", "regionnortheast", "regionmidwest", "regionwest",
         "gendermale", "marriedyes", "income", "school", "employedyes")
X1 <- X[, focus]
X2 <- X[, aux]
y <- NMES1988$visits

# starting values from glm.nb() from MASS
startFit <- MASS::glm.nb(y ~ X[,-1])
betaStart <- coef(startFit)
rhoStart <- startFit$theta
k1 <- ncol(X1)
k2 <- ncol(X2)

str(walsNBfit(X1, X2, y, rhoStart, family = negbinWALS(scale = rhoStart, link = "log"),
              betaStart1 = betaStart[1:k1],
              betaStart2 = betaStart[(k1 + 1):(k1 + k2)],
              prior = weibull(), method = "fullSVD"))

}
\references{
\insertAllCited{}
}
\seealso{
\link{walsNB}, \link{walsNBfitIterate}.
}
