#' Saves the plots corresponding to the TailID detection
#'
#' This function saves the plots corresponding to the TailID detection, which includes: targeted candidates plot, shape variation plot, and inconsistent detected points.
#'
#' @param output_dir Path to save the plots.
#' @param sample A numeric vector.
#' @param pm_max A number between 0 and 1 indicating the threshold of maximum extreme values to consider.
#' @param pm_min A number between 0 and 1 indicating the threshold of minimum extreme values to consider.
#' @param pc_max A number between pm_max and 1 indicating the threshold of maximum sensitive points to consider.
#' @param pc_min A number between pm_min and 1 indicating the threshold of minimum sensitive points to consider.
#' @param conf_level A number between 0 and 1 indicating the confidence level for the detection.
#'
#' @return A vector of indices corresponding to the detected sensitive points.
#' @examples
#' output_dir <- file.path(tempdir(), "output")
#' if (dir.exists(output_dir) || dir.create(output_dir, recursive = TRUE)) {
#'   plot_TailID(output_dir, rnorm(1000), 0.85, 0.85, 0.999, 0.999, 0.95)
#' }
#' if (dir.exists(output_dir) || dir.create(output_dir, recursive = TRUE)) {
#'   plot_TailID(output_dir, c(rnorm(10^3, 10, 1), rnorm(10, 20, 3)), 0.85, 0.85, 0.99, 0.99, 0.99999)
#' }
#' @importFrom stats na.omit quantile
#' @export

plot_TailID <- function(output_dir, sample, pm_max, pm_min, pc_max, pc_min, conf_level) {

  # Select candidate points for evaluation
  candidates <- candidate_selection(sample, pc_max, pc_min)
  candidates_max <- candidates[[1]]
  candidates_min <- candidates[[2]]
  candidates <- unlist(candidates)

  # Define colors and shapes for the plots
  colors <- grDevices::hcl.colors(length(candidates_max), "plasma")
  colors <- sample(colors)
  valid_shapes <- c(0, 1, 2, 5, 6, 15, 16, 17, 18, 19, 20)
  shapes <- rep(valid_shapes, length.out = length(candidates_max))

  # Define colors and shapes for the plots
  colors_min <- grDevices::hcl.colors(length(candidates_min), "plasma")
  colors_min <- sample(colors_min)
  valid_shapes_min <- c(0, 1, 2, 5, 6, 15, 16, 17, 18, 19, 20)
  shapes_min <- rep(valid_shapes_min, length.out = length(candidates_min))

  # Create a data frame for plotting candidates
  df <- data.frame(
    index = seq_along(sample),
    value = sample,
    sensitive = ifelse(seq_along(sample) %in% candidates, "Sensitive", "Normal"),
    color = rep("black", length(sample)),
    shape = 16
  )

  # Assign specific colors and shapes to sensitive points
  df$color[candidates_max] <- colors
  df$shape[candidates_max] <- shapes
  df$color[candidates_min] <- colors_min
  df$shape[candidates_min] <- shapes_min

  # Set up breaks for the y-axis
  range_min <- min(df$value, na.rm = TRUE)
  range_max <- max(df$value, na.rm = TRUE)

  mid_breaks <- seq(
    range_min + (range_max - range_min) * 0.2,
    range_max - (range_max - range_min) * 0.2,
    length.out = 2
  )

  shapes <- unique(shapes)

  # Generate a scatter plot for candidate points
  gg <- ggplot2::ggplot(df, ggplot2::aes(x = index, y = value)) +
    ggplot2::geom_point(ggplot2::aes(
      color = color,
      shape = factor(shape, levels = shapes),  # Enforce specific order
      size = ifelse(sensitive == "Sensitive", 1, 0.1)
    )) +
    ggplot2::scale_color_identity() +
    ggplot2::scale_shape_manual(values = shapes) +  # Use explicit shape mapping
    ggplot2::scale_size_identity() +
    ggplot2::scale_y_continuous(
      labels = scales::scientific,
      breaks = mid_breaks
    ) +
    ggplot2::labs(x = "Observation Index", y = "Observation Value") +
    ggplot2::theme_bw() +
    ggplot2::theme(
      legend.position = "none",
      axis.text.y = ggplot2::element_text(angle = 90, hjust = 0.5)
    )

  print(gg)

  pdf_path <- file.path(output_dir, "candidates.pdf")
  # Save the scatter plot
  ggplot2::ggsave(filename = pdf_path, plot = gg, width = 3.5, height = 3)
  candidates <- list(candidates_max, candidates_min)
  # Evaluate shape parameters and confidence intervals
  information <- shape_evaluation(sample, candidates, pm_max, pm_min, conf_level)
  shape_max <- information[[1]]
  shape_min <- information[[2]]
  int_conf_max <- information[[3]]
  int_conf_min <- information[[4]]
  sensitive_points <- information[[5]]

  shapes <- rep(valid_shapes, length.out = length(candidates_max))

  # Plot shape variation for maximums
  plot_data <- data.frame(
    index = seq_along(shape_max[2:length(shape_max)]),
    parameter = unlist(shape_max[2:length(shape_max)]),
    color = colors[length(colors) - (seq_along(shape_max[2:length(shape_max)]) -1)],
    shape = shapes[length(shapes) - (seq_along(shape_max[2:length(shape_max)]) - 1)]
  )

  plot_data$index <- as.numeric(plot_data$index)

  shapes <- unique(shapes)

  gg <- ggplot2::ggplot(plot_data, ggplot2::aes(
    x = index,
    y = parameter,
    color = color,
    shape = factor(shape, levels = shapes)  # Ensure the order is preserved
  )) +
    ggplot2::geom_hline(yintercept = shape_max[[1]], color = "black", alpha = 0.25) +
    ggplot2::geom_hline(yintercept = int_conf_max[2], linetype = "dashed", color = "black", alpha = 0.25) +
    ggplot2::scale_color_identity() +
    ggplot2::scale_shape_manual(values = shapes) +  # Use explicit shape order
    ggplot2::theme_bw() +
    ggplot2::theme(legend.position = "none") +
    ggplot2::labs(x = "Number of Candidates", y = "GPD Shape")


  # Add confidence interval lines for max shape
  count <- 1
  for (a in seq_along(shape_max[2:length(shape_max)])) {
    if (!is.na(shape_max[[a]])) {
      gg <- gg +
        ggplot2::geom_hline(
          yintercept = int_conf_max[2 + count + 1],
          linetype = "dashed",
          color = colors[length(colors) - (a - 1)]
        )
      count <- count + 2
    }
  }

  gg <- gg + ggplot2::geom_point(size = 1, na.rm = TRUE)
  print(gg)
  pdf_path <- file.path(output_dir, "shape_max.pdf")
  ggplot2::ggsave(file = pdf_path, plot = gg, width = 3.5, height = 3)

  shapes_min <- rep(valid_shapes, length.out = length(candidates_min))

  # Repeat for minimum shape variation
  # Similar setup as for maximums, with adjustments for `shape_min`
  plot_data <- data.frame(
    index = seq_along(shape_min[2:length(shape_min)]),
    parameter = unlist(shape_min[2:length(shape_min)]),
    color = colors_min[length(colors_min) - (seq_along(shape_min[2:length(shape_min)]) - 1)],
    shape = shapes_min[length(shapes_min) - (seq_along(shape_min[2:length(shape_min)]) - 1)]
  )
  plot_data$index <- as.numeric(plot_data$index)

  shapes_min <- unique(shapes_min)

  gg <- ggplot2::ggplot(plot_data, ggplot2::aes(
    x = index,
    y = parameter,
    color = color,
    shape = factor(shape, levels = shapes_min)  # Ensure the correct order
  )) +
    ggplot2::geom_hline(yintercept = shape_min[[1]], color = "black", alpha = 0.25) +
    ggplot2::geom_hline(yintercept = int_conf_min[2], linetype = "dashed", color = "black", alpha = 0.25) +
    ggplot2::scale_color_identity() +
    ggplot2::scale_shape_manual(values = shapes_min) +  # Use predefined shape order
    ggplot2::theme_bw() +
    ggplot2::theme(legend.position = "none") +
    ggplot2::labs(x = "Number of Candidates", y = "GPD Shape")


  count <- 1
  for (a in seq_along(shape_min[2:length(shape_min)])) {
    if (!is.na(shape_min[[a]])) {
      gg <- gg +
        ggplot2::geom_hline(
          yintercept = int_conf_min[2 + count + 1],
          linetype = "dashed",
          color = colors_min[length(colors_min) - (a - 1)]
        )
      count <- count + 2
    }
  }

  gg <- gg + ggplot2::geom_point(size = 1, na.rm = TRUE)
  print(gg)
  pdf_path <- file.path(output_dir, "shape_min.pdf")
  ggplot2::ggsave(file = pdf_path, plot = gg, width = 3.5, height = 3)

  # Highlight sensitive points in a final detection plot
  df <- data.frame(
    index = seq_along(sample),
    value = sample,
    type = "Normal"
  )
  df$type[unlist(candidates)] <- "Sensitive"

  gg <- ggplot2::ggplot(df, ggplot2::aes(x = index, y = value)) +
    ggplot2::geom_point(size = 0.1, color = "black") +
    ggplot2::labs(title = "Sensitive Points Detected", x = "Observation Index", y = "Observation Value") +
    ggplot2::theme_bw() +
    ggplot2::geom_point(data = df[unlist(candidates), ], ggplot2::aes(x = index, y = value), color = "black", size = 0.5)

  if (length(sensitive_points) > 0) {
    gg <- gg + ggplot2::geom_point(data = df[sensitive_points, ], ggplot2::aes(x = index, y = value), color = "red", size = 0.5)
  }
  print(gg)
  pdf_path <- file.path(output_dir, "detection.pdf")
  ggplot2::ggsave(file = pdf_path, plot = gg, width = 3.5, height = 3)
}

