# Dummy dataset for testing plot_SynergyLMM ----
set.seed(123)
test_data <- data.frame(
  SampleID = rep(1:10, each = 10),
  Time = rep(0:9, times = 10),
  Treatment = rep(
    c("Control", "Drug_A", "Drug_B", "Drug_AB"),
    each = 10,
    length.out = 100
  ),
  TV = rnorm(100, mean = 100, sd = 20)
)

# Model for testing ----

lmm <- lmmModel(
  data = test_data,
  sample_id = "SampleID",
  time = "Time",
  treatment = "Treatment",
  tumor_vol = "TV",
  trt_control = "Control",
  drug_a = "Drug_A",
  drug_b = "Drug_B",
  combination = "Drug_AB",
  time_start = 0,
  time_end = NULL,
  min_observations = 1,
  show_plot = FALSE
)

# Synergy results for testing ----

lmmSyn <- lmmSynergy(lmm, show_plot = FALSE)

test_that("plot_lmmSynergy runs without error on valid input", {
  expect_true(ggplot2::is_ggplot(
    plot_SynergyLMM(
      lmm,
      plot_type = "lmmModel",
      trt_control = "Control",
      drug_a = "Drug_A",
      drug_b = "Drug_B",
      combination = "Drug_AB"
    )
  ))
  expect_equal(length(plot_SynergyLMM(lmmSyn, plot_type = "lmmSynergy")),3)
  expect_equal(length(plot_SynergyLMM(lmm, "ranefDiagnostics")), 2)
  expect_equal(length(plot_SynergyLMM(lmm, "ranefDiagnostics")$plots), 4)
  expect_equal(length(plot_SynergyLMM(lmm, "residDiagnostics")), 6)
  expect_equal(length(plot_SynergyLMM(lmm, "ObsvsPred")), 45)
})

test_that("Input validation works", {
  invalid_obj <- list()
  expect_error(
    plot_SynergyLMM(invalid_obj, "lmmModel"),
    "Incorrect input object. Input object needs to be obtained by 'lmmModel' or 'lmmSynergy' functions."
  )
  
  expect_error(
    plot_SynergyLMM(lmmSyn, "lmmModel"),
    "The input object needs to be a model generated by 'lmmModel' function if 'plot_type = lmmModel' is selected."
  )
  
  expect_error(
    plot_SynergyLMM(lmm, "lmmSynergy"),
    "The input object needs to be a model generated by 'lmmSynergy' function if 'plot_type = lmmSynergy' is selected."
  )
})

test_that("Invalid plot_type raises an error", {
  expect_error(
    plot_SynergyLMM(lmm, "InvalidType"),
    "The 'plot_type' value is not a valid type of plot."
  )
})

test_that("Invalid arguments raises error when plot_type = lmmModel", {
  expect_error(
    plot_SynergyLMM(lmm, plot_type = "lmmModel"),
    "Treatment group names provided do not coincide with treatment group names in the model. Please, provide the correct treatment group names in the arguments."
  )
})
