% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reilly.R
\name{reilly}
\alias{reilly}
\title{Reilly Catchment Areas}
\usage{
reilly(
  knownpts,
  unknownpts,
  matdist,
  varname,
  typefct = "exponential",
  span,
  beta,
  resolution,
  mask,
  bypassctrl = FALSE,
  longlat = TRUE,
  returnclass = "sp"
)
}
\arguments{
\item{knownpts}{sp or sf object; 
this is the set of known observations to estimate the catchment areas from.}

\item{unknownpts}{sp or sf object; 
this is the set of unknown units for which the function computes the estimates. 
Not used when \code{resolution} is set up. (optional)}

\item{matdist}{matrix; distance matrix between known observations and unknown 
units for which the function computes the estimates. Row names match the row 
names of \code{knownpts} and column names match the row names of 
\code{unknownpts}. \code{matdist} can contain any distance metric (time 
distance or euclidean distance for example). If \code{matdist} is not set, 
the distance matrix is built with \code{\link{CreateDistMatrix}}. (optional)}

\item{varname}{character; name of the variable in the \code{knownpts} dataframe 
from which values are computed. Quantitative variable with no negative values.}

\item{typefct}{character; spatial interaction function. Options are "pareto" 
(means power law) or "exponential".
If "pareto" the interaction is defined as: (1 + alpha * mDistance) ^ (-beta).
If "exponential" the interaction is defined as: 
exp(- alpha * mDistance ^ beta).
The alpha parameter is computed from parameters given by the user 
(\code{beta} and \code{span}).}

\item{span}{numeric; distance where the density of probability of the spatial 
interaction function equals 0.5.}

\item{beta}{numeric; impedance factor for the spatial interaction function.}

\item{resolution}{numeric; resolution of the output grid (in map units). 
If resolution is not set, the grid will contain around 7250 points. (optional)}

\item{mask}{sp or sf object; the spatial extent of this object is used to 
create the regularly spaced points output. (optional)}

\item{bypassctrl}{logical; bypass the distance matrix size control (see 
\code{\link{CreateDistMatrix}} Details).}

\item{longlat}{logical; if FALSE, Euclidean distance, if TRUE Great Circle 
(WGS84 ellipsoid) distance.}

\item{returnclass}{"sp" or "sf"; class of the returned object.}
}
\value{
Point object with the computed catchment areas in a new 
field named \code{OUTPUT}. Values match the row names of \code{knownpts}.
}
\description{
This function computes the catchment areas as defined by W.J. Reilly (1931).
}
\examples{
# Create a grid of paris extent and 200 meters
# resolution
data(hospital)
mygrid <- CreateGrid(w = hospital, resolution = 200, returnclass = "sf")
# Create a distance matrix between known points (hospital) and mygrid
mymat <- CreateDistMatrix(knownpts = hospital, unknownpts = mygrid)
# Compute Reilly catchment areas from known points (hospital) on a given
# grid (mygrid) using a given distance matrix (mymat)
myreilly2 <- reilly(knownpts = hospital, unknownpts = mygrid,
                    matdist = mymat, varname = "capacity",
                    typefct = "exponential", span = 1250,
                    beta = 3, mask = paris, returnclass = "sf")
# Compute Reilly catchment areas from known points (hospital) on a
# grid defined by its resolution
myreilly <- reilly(knownpts = hospital, varname = "capacity",
                   typefct = "exponential", span = 1250, beta = 3,
                   resolution = 200, mask = paris, returnclass = "sf")
# The function output an sf object
class(myreilly)
# The OUTPUT field values match knownpts row names
head(unique(myreilly$OUTPUT))
}
\references{
REILLY, W. J. (1931) The law of retail gravitation, W. J. Reilly, New York.
}
\seealso{
\link{reilly}, \link{rasterReilly}, \link{plotReilly}, \link{CreateGrid}, 
\link{CreateDistMatrix}.
}
