#' Sample size calculator for estimation of population proportion under SRS
#'
#' @description The SscSrsProp function calculates the sample size required for estimation of population proportion under Simple Random Sampling (SRS) with or without replacement sampling design for a given confidence level and relative error under the assumption than population CV is unknown. Further, the function provides the number of additional sample units to be surveyed over the preliminary sample under both SRS with or without replacement.
#'
#' @param replace replace=TRUE, if sampling design is SRSWR and replace=FALSE, if sampling design is SRSWOR.
#' @param alpha Level of significance value, alpha=0.01 at 1 percent level of significance and alpha=0.05 at 5 percent level of significance.
#' @param re Relative error in the estimation of population proportion (e.g. 0.1, 0.5).
#' @param N Population size. When sampling design is SRSWR, then write NA.
#' @param preliminary_sample Values of the study variable (in 0 or 1) for the preliminary sample.
#'
#' @return A list with the following components:
#' \item{Required sample size}{Sample size required for estimation of population proportion.}
#' \item{Additional sample units to be surveyed}{Additional sample units to be surveyed over the preliminary sample for estimation of population proportion. If the value of additional sample units to be surveyed is negative, then preliminary sample is considered as the final sample.}
#'
#' @details This function returns the required sample size as well as additional sample units to be surveyed over the preliminary sample for both SRS with or without replacement sampling design.
#'
#' @references
#' Cochran, W. G. (1977). \emph{Sampling Techniques, 3rd Edition}. New York: John Wiley & Sons, Inc.
#'
#' Singh, D. and Chaudhary, F.S. (1986). \emph{Theory and Analysis of Sample Survey Designs}. New York: John Wiley & Sons, Inc.
#'
#' Sukhatme, P.V., Sukhatme, B.V., Sukhatme, S. and Asok, C. (1984). \emph{Sampling Theory of Surveys with Applications}. Iowa State University Press, Ames and Indian Society of Agricultural Statistics, New Delhi.
#'
#' @examples
#' # Calculate sample size for SRSWOR design for estimation of population proportion
#' preliminary_sample=c(1,0,1,1,1,0,0,0,1,1)
#' SscSrsProp(FALSE, 0.05, 0.2, 500, preliminary_sample)
#' # Calculate sample size for SRSWR design for estimation of population proportion
#' preliminary_sample=c(1,0,1,1,0,1,1,1,0,0,0,1,1,1,1)
#' SscSrsProp(TRUE, 0.05, 0.2, NA, preliminary_sample)
#' @export
#' @importFrom stats qt
SscSrsProp=function(replace,alpha,re,N,preliminary_sample)
{
  p1=sum(preliminary_sample)/length(preliminary_sample)
  q1=1-p1
  t=qt(alpha/2,(length(preliminary_sample)-1),lower.tail = FALSE)
  # Calculate sample size for SRSWOR design for estimation of population proportion
  if(replace==FALSE)
  {
    size=round(N*(t^2*q1+re^2*p1)/(t^2*q1+re^2*p1*N),0)
    additional_size=size-length(preliminary_sample)
    output <- list("Required sample size"=size, "Additional sample units to be surveyed"=additional_size)
    return(output)
  }
  #Calculate sample size for SRSWR design for estimation of population proportion
  else if(replace==TRUE)
  {
    size=round((t^2*q1+re^2*p1)/(re^2*p1),0)
    additional_size=size-length(preliminary_sample)
    output <- list("Required sample size"=size, "Additional sample units to be surveyed"=additional_size)
    return(output)
  }
}
