% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unsuperClass.R
\name{unsuperClass}
\alias{unsuperClass}
\title{Unsupervised Classification}
\usage{
unsuperClass(
  img,
  nSamples = 10000,
  nClasses = 5,
  nStarts = 25,
  nIter = 100,
  norm = FALSE,
  clusterMap = TRUE,
  algorithm = "Hartigan-Wong",
  output = "classes",
  ...
)
}
\arguments{
\item{img}{SpatRaster.}

\item{nSamples}{Integer. Number of random samples to draw to fit cluster map. Only relevant if clusterMap = TRUE.}

\item{nClasses}{Integer. Number of classes.}

\item{nStarts}{Integer. Number of random starts for kmeans algorithm.}

\item{nIter}{Integer. Maximal number of iterations allowed.}

\item{norm}{Logical. If \code{TRUE} will normalize img first using \link{normImage}. Normalizing is beneficial if your predictors have different scales.}

\item{clusterMap}{Logical. Fit kmeans model to a random subset of the img (see Details).}

\item{algorithm}{Character. \link[stats]{kmeans} algorithm. One of c("Hartigan-Wong", "Lloyd", "MacQueen")}

\item{output}{Character. Either 'classes' (kmeans class; default) or 'distances' (euclidean distance to each cluster center).}

\item{...}{further arguments to be passed to \link[terra]{writeRaster}, e.g. filename}
}
\value{
Returns an RStoolbox::unsuperClass object, which is a list containing the kmeans model ($model) and the raster map ($map).
For output = "classes", $map contains a SpatRaster with discrete classes (kmeans clusters); for output = "distances" $map contains
a SpatRaster, with `nClasses` layers, where each layer maps the euclidean distance to the corresponding class centroid.
}
\description{
Unsupervised clustering of SpatRaster data using kmeans clustering
}
\details{
Clustering is done using \code{\link[stats]{kmeans}}. This can be done for all pixels of the image (\code{clusterMap=FALSE}), however this can be slow and is
not memory safe. Therefore if you have large raster data (> memory), as is typically the case with remote sensing imagery it is advisable to choose clusterMap=TRUE (the default).
This means that a kmeans cluster model is calculated based on a random subset of pixels (\code{nSamples}). Then the distance of *all* pixels to the cluster centers 
is calculated in a stepwise fashion using \code{\link[terra]{predict}}. Class assignment is based on minimum euclidean distance to the cluster centers.

The solution of the kmeans algorithm often depends on the initial configuration of class centers which is chosen randomly. 
Therefore, kmeans is usually run with multiple random starting configurations in order to find a convergent solution from different starting configurations.
The \code{nStarts} argument allows to specify how many random starts are conducted.
}
\examples{
\dontrun{
library(terra)
input <- rlogo

## Plot 
olpar <- par(no.readonly = TRUE) # back-up par
par(mfrow=c(1,2))
plotRGB(input)

## Run classification
set.seed(25)
unC <- unsuperClass(input, nSamples = 100, nClasses = 5, nStarts = 5)
unC

## Plots
colors <- rainbow(5)
plot(unC$map, col = colors, legend = FALSE, axes = FALSE, box = FALSE)
legend(1,1, legend = paste0("C",1:5), fill = colors, title = "Classes", horiz = TRUE,  bty = "n")

## Return the distance of each pixel to each class centroid
unC <- unsuperClass(input, nSamples = 100, nClasses = 3, output = "distances")
unC

ggR(unC$map, 1:3, geom_raster = TRUE)

par(olpar) # reset par
}
}
