% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcConnectivity.R
\name{reverseTransition}
\alias{reverseTransition}
\alias{reversePsiRelAbund}
\alias{reverseTransitionRelAbund}
\alias{reversePi}
\title{Reverse transition probabilities and origin relative abundance}
\usage{
reverseTransition(
  psi = NULL,
  originRelAbund = NULL,
  pi = NULL,
  originSites = NULL,
  targetSites = NULL,
  originNames = NULL,
  targetNames = NULL,
  nSamples = 1000,
  row0 = 0,
  alpha = 0.05
)

reversePsiRelAbund(
  psi = NULL,
  originRelAbund = NULL,
  pi = NULL,
  originSites = NULL,
  targetSites = NULL,
  originNames = NULL,
  targetNames = NULL,
  nSamples = 1000,
  row0 = 0,
  alpha = 0.05
)

reverseTransitionRelAbund(
  psi = NULL,
  originRelAbund = NULL,
  pi = NULL,
  originSites = NULL,
  targetSites = NULL,
  originNames = NULL,
  targetNames = NULL,
  nSamples = 1000,
  row0 = 0,
  alpha = 0.05
)

reversePi(
  psi = NULL,
  originRelAbund = NULL,
  pi = NULL,
  originSites = NULL,
  targetSites = NULL,
  originNames = NULL,
  targetNames = NULL,
  nSamples = 1000,
  row0 = 0,
  alpha = 0.05
)
}
\arguments{
\item{psi}{Transition probabilities between B origin and W target sites.
Either a matrix with B rows and W columns where rows sum to 1, an array with
dimensions x, B, and W (with x samples of the transition probability matrix
from another model), an 'estPsi' object (result of calling estTransition),
or a MARK object with estimates of transition probabilities}

\item{originRelAbund}{Relative abundance estimates at B origin sites. Either
a numeric vector of length B that sums to 1 or an mcmc object with at least
\code{nSamples} rows and columns including 'relN[1]' through 'relN[B]'}

\item{pi}{Migratory combination (joint) probabilities. Either a matrix with B
rows and W columns where all entries sum to 1, an array with dimensions x,
B, and W, or an 'estPi' object (currently only the results of calling this
function) Either pi or psi and originRelAbund should be specified.}

\item{originSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are origin}

\item{targetSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are target}

\item{originNames}{Vector of names for the origin sites. If not provided, the
function will try to get them from psi}

\item{targetNames}{Vector of names for the target sites. If not provided, the
function will try to get them from psi}

\item{nSamples}{Number of times to resample \code{psi} and/or
\code{originRelAbund}. The purpose is to estimate sampling uncertainty;
higher values here will do so with more precision}

\item{row0}{If \code{originRelAbund} is an mcmc object or array, this can be
set to 0 (default) or any greater integer to specify where to stop ignoring
samples (additional "burn-in")}

\item{alpha}{Level for confidence/credible intervals provided. Default (0.05)
gives 95 percent CI}
}
\value{
If both psi and originRelAbund are simple point estimates,
\code{reversePsiRelAbund} returns a list with point estimates of gamma,
targetRelAbund, and pi. Otherwise, it returns a list with the elements:
\describe{
  \item{\code{gamma}}{List containing estimates of reverse transition
  probabilities:
  \itemize{
   \item{\code{sample}} Array of sampled values for gamma. \code{nSamples} x
     [number of target sites] x [number of origin sites]. Provided to allow
     the user to compute own summary statistics.
   \item{\code{mean}} Main estimate of gamma matrix. [number of target sites]
     x [number of origin sites].
   \item{\code{se}} Standard error of gamma, estimated from SD of
     \code{gamma$sample}.
   \item{\code{simpleCI}} \code{1 - alpha} confidence interval for gamma,
     estimated as \code{alpha/2} and \code{1 - alpha/2} quantiles of
     \code{gamma$sample}.
   \item{\code{bcCI}} Bias-corrected \code{1 - alpha} confidence interval
     for gamma. May be preferable to \code{simpleCI} when \code{mean} is the
     best estimate of gamma. \code{simpleCI} is preferred when
     \code{median} is a better estimator. When the mean and median are equal,
     these should be identical.  Estimated as the
     \code{pnorm(2 * z0 + qnorm(alpha / 2))} and
     \code{pnorm(2 * z0 + qnorm(1 - alpha / 2))} quantiles of \code{sample},
     where z0 is the proportion of \code{sample < mean}.
   \item{\code{median}} Median estimate of gamma matrix.
   \item{\code{point}} Simple point estimate of gamma matrix, not accounting
     for sampling error.
  }
  }
  \item{\code{targetRelAbund}}{List containing estimates of relative
   abundance at target sites. Items within are the same as within gamma,
   except for having one fewer dimension.}
  \item{\code{pi}}{List containing estimates of origin/target site
   combination probabilities (sum to 1). Items within are the same as within
   gamma, except for reversing dimensions (same order as psi).}
  \item{\code{input}}{List containing the inputs to \code{reversePsiRelAbund}.}
}
If the input is pi instead of psi and originRelAbund, then pi is not an
output, but psi and originRelAbund are. Otherwise the same.
}
\description{
Reverse transition probabilities (psi; sum to 1 for each origin site) and
origin relative abundance (originRelAbund; sum to 1 overall) estimates to
calculate or estimate target site to origin site transition probabilities
(gamma; sum to 1 for each target site), target site relative abundances
(targetRelAbund; sum to 1 overall), and origin/target site combination
probabilities (pi; sum to 1 overall). If either psi or originRelAbund is an
estimate with sampling uncertainty expressed, this function will propagate
that uncertainty to provide true estimates of gamma, targetRelAbund, and pi;
otherwise (if both are simple point estimates), it will also provide point
estimates.
}
\details{
Alternatively, can be used to reverse migratory combination (joint)
probabilities (pi; sum to 1 overall) to psi, originRelAbund, gamma, and
targetRelAbund.
}
\examples{
## Example 1: sample psis and relative abundances from Cohen et al. (2018)
## (no uncertainty in psi or relative abundance)
for (i in 1:length(samplePsis)) {
 for (j in 1:length(sampleOriginRelN)){
  cat("For psi:\n")
  print(samplePsis[[i]])
  cat("and origin relative abundance:", sampleOriginRelN[[j]], "\n")
  print(reverseTransition(samplePsis[[i]], sampleOriginRelN[[j]]))
 }
}

## Example 2: Common tern banding example (uncertainty in psi, not relative
## abundance)
# Number of MCMC iterations
ni. <- 1000 # reduced from 70000 for example speed
# Number of iterations to thin from posterior
nt. <- 1
# Number of iterations to discard as burn-in
nb. <- 500 # reduced from 20000 for example speed
# Number of MCMC chains
nc. <- 1 # reduced from 3 for example speed
COTE_banded <- c(10360, 1787, 2495, 336)
COTE_reencountered <- matrix(c(12, 0, 38, 15,
                               111, 7, 6, 2,
                               5, 0, 19, 4,
                               1123, 40, 41, 7),
                             4, 4,
                             dimnames = list(LETTERS[1:4], 1:4))
COTE_psi <- estTransition(originNames = LETTERS[1:4],
                          targetNames = 1:4,
                          banded = COTE_banded,
                          reencountered = COTE_reencountered,
                          verbose = 1,
                          nSamples = (ni. - nb.) / nt. * nc., nBurnin = nb.,
                          nThin = nt., nChains = nc.,
                          method = "MCMC")
COTE_psi
COTE_rev <- reverseTransition(COTE_psi, sampleOriginRelN[[1]],
                               nSamples = 2000)
COTE_rev

\donttest{
## Example 3: Uncertainty in both psi and relative abundance
# Number of populations
nOriginSites <- 3; originNames <- LETTERS[1:nOriginSites]
nTargetSites <- 4; targetNames <- 1:nTargetSites

originRelAbund <- c(1/3, 1/3, 1/3)

psiTrue <- array(0, c(nOriginSites, nTargetSites),
                 list(originNames, targetNames))
psiTrue[1,] <- c(0.22, 0.52, 0.16, 0.10)
psiTrue[2,] <- c(0.41, 0.31, 0.17, 0.11)
psiTrue[3,] <- c(0.10, 0.15, 0.42, 0.33)
rowSums(psiTrue)

rev <- reverseTransition(psiTrue, originRelAbund)

# Simulate abundance data on origin sites
# Number of routes w/i each population (assumed to be balanced)
routePerPop. <- 30 # reduced for example speed
# Number of years
nYears. <- 5 # reduced for example speed
# log(Expected number of birds counted at each route)
alphaPop. <- 1.95
# standard deviation of normal distribution assumed for route/observer random
# effects
sdRoute. <- 0.6
# standard deviation of normal distribution assumed for year random effects
sdYear. <- 0.18
# Number of MCMC iterations
ni. <- 1000 # reduced from 70000 for example speed
# Number of iterations to thin from posterior
nt. <- 1
# Number of iterations to discard as burn-in
nb. <- 500 # reduced from 20000 for example speed
# Number of MCMC chains
nc. <- 1 # reduced from 3 for example speed

sim_data <- simCountData(nStrata = nOriginSites, routesPerStrata = routePerPop.,
                         nYears = nYears., alphaStrat = alphaPop.,
                         sdRoute = sdRoute., sdYear = sdYear.)
# Estimate population-level abundance
out_mcmc <- modelCountDataJAGS(count_data = sim_data, ni = ni., nt = nt.,
                               nb = nb., nc = nc.)

# Simulate movement data
sampleSize <- list(rep(20, nOriginSites), NULL)
captured <- rep("origin", sum(sampleSize[[1]]))
isTelemetry <- rep(TRUE:FALSE, c(sum(sampleSize[[1]]), sum(sampleSize[[2]])))
isProb <- rep(FALSE:TRUE, c(sum(sampleSize[[1]]), sum(sampleSize[[2]])))

# Telemetry data (released origin)
data1 <- simTelemetryData(psi = psiTrue,
                          sampleSize = sampleSize[[1]],
                          captured = "origin")
tt <- data1$targetAssignment
oa <- data1$originAssignment

# Estimate transition probabilities (psi)
est1 <- estTransition(targetAssignment = tt,
                      originAssignment = oa,
                      originNames = originNames,
                      targetNames = targetNames,
                      nSamples = 500, isGL = FALSE,
                      isTelemetry = isTelemetry,
                      isRaster = FALSE,
                      isProb = isProb,
                      captured = captured,
                      nSim = 10, verbose = 0)
# Reverse estimates
rev1 <- reverseTransition(psi = est1, originRelAbund = out_mcmc)
# Compare estimates of gamma, target relative abundance, and pi with calculation
# from true values
rev
rev1
}
}
