% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estConnectivity.R
\name{estNMC}
\alias{estNMC}
\title{Estimate NMC_XY, another type of migratory connectivity strength}
\usage{
estNMC(
  psi,
  originRelAbund = NULL,
  originNames = NULL,
  targetNames = NULL,
  originSites = NULL,
  targetSites = NULL,
  nSamples = 1000,
  row0 = 0,
  verbose = 0,
  alpha = 0.05,
  returnAllInput = TRUE
)
}
\arguments{
\item{psi}{Transition probabilities between X origin and Y target
sites/nodes. Either an array with dimensions n, X, and Y (with n samples of
the transition probability matrix from another model), an 'estPsi' object
(result of calling estTransition), or a MARK object with estimates of
transition probabilities}

\item{originRelAbund}{Optional. Relative abundance estimates at X origin
sites (nodes). Either a numeric vector of length X that sums to 1, or an
mcmc object (such as is produced by \code{\link{modelCountDataJAGS}}) or
matrix with at least \code{nSamples} rows. If there are more than X columns,
the relevant columns should be labeled "relN[1]" through "relN[X]"}

\item{originNames}{Optional. Vector of names for the origin sites. Mostly for
internal use}

\item{targetNames}{Optional. Vector of names for the target sites. Mostly for
internal use}

\item{originSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are origin}

\item{targetSites}{If \code{psi} is a MARK object, this must be a numeric
vector indicating which sites are target}

\item{nSamples}{Number of times to resample \code{psi}. The purpose is to
estimate sampling uncertainty; higher values here will do so with more
precision}

\item{row0}{If \code{originRelAbund} is an mcmc object or array, this can be
set to 0 (default) or any greater integer to specify where to stop ignoring
samples ("burn-in")}

\item{verbose}{0 (default) to 2. 0 prints no output during run. 1 prints
a progress update and summary every 100 samples. 2 prints a
progress update and summary every sample}

\item{alpha}{Level for confidence/credible intervals provided. Default (0.05)
gives 95 percent CI}

\item{returnAllInput}{if TRUE (the default) the output includes all of the
inputs. If FALSE, only the inputs currently used by another MigConnectivity
function are included in the output to save memory.}
}
\value{
\code{estNMC} returns a list with the elements:
\describe{
  \item{\code{NMC}}{List containing estimates of network migratory
   connectivity strength:
   \itemize{
     \item{\code{sample}} \code{nSamples} sampled values for
      NMC_XY. Provided to allow the user to compute own summary statistics.
     \item{\code{mean}} Mean of \code{NMC$sample}. Main estimate of NMC_XY,
      incorporating parametric uncertainty.
     \item{\code{se}} Standard error of NMC, estimated from SD of
      \code{NMC$sample}.
     \item{\code{simpleCI}} Default\code{1 - alpha} confidence interval for
      NMC_XY, estimated as \code{alpha/2} and \code{1 - alpha/2} quantiles of
      \code{NMC$sample}.
     \item{\code{bcCI}} Bias-corrected \code{1 - alpha} confidence interval
      for NMC_XY. May be preferable to \code{NMC$simpleCI} when
      \code{NMC$mean} is the best estimate of NMC_XY. \code{NMC$simpleCI} is
      preferred when \code{NMC$median} is a better estimator. When
      \code{NMC$mean==NMC$median}, these should be identical.  Estimated as
      the \code{pnorm(2 * z0 + qnorm(alpha / 2))} and
      \code{pnorm(2 * z0 + qnorm(1 - alpha / 2))} quantiles of
      \code{NMC$sample}, where z0 is the proportion of
      \code{NMC$sample < NMC$mean}.
     \item{\code{hpdCI}} \code{1 - alpha} credible interval for NMC,
      estimated using the highest posterior density (HPD) method.
     \item{\code{median}} Median of NMC_XY, alternate point estimate also
      including parametric uncertainty.
     \item{\code{point}} Simple point estimate of NMC_XY, using the point
      estimate of \code{psi} (usually the mean values), not accounting for
      sampling error.
   }
  }
  \item{\code{NMCpop}}{List containing estimates of network migratory
   connectivity diversity (X node-specific NMC_XY):
   \itemize{
     \item{\code{sample}} Matrix of sampled values for network migratory
      connectivity diversity. \code{nSamples} x [number of origin sites].
      Provided to allow the user to compute own summary statistics.
     \item{\code{mean}} Column means of \code{NMCpop$sample}. Main estimate
      of network migratory connectivity diversity, incorporating parametric
      uncertainty.
     \item{\code{se}} Standard errors of network migratory
      connectivity diversity, estimated from column
      standard deviations of \code{NMCpop$sample}.
     \item{\code{simpleCI}} Default\code{1 - alpha} confidence interval for
      network migratory connectivity diversity, estimated as \code{alpha/2}
      and \code{1 - alpha/2} quantiles of each column of \code{NMCpop$sample}.
     \item{\code{bcCI}} Bias-corrected \code{1 - alpha} confidence intervals
      for network migratory connectivity diversity. May be preferable to
      \code{NMCpop$simpleCI} when \code{NMCpop$mean} are the best estimate of
      network migratory connectivity diversity. \code{NMCpop$simpleCI} is
      preferred when \code{NMCpop$median} is a better estimator. When
      \code{NMCpop$mean==NMCpop$median}, these should be identical.
      Estimated as the \code{pnorm(2 * z0 + qnorm(alpha / 2))} and
      \code{pnorm(2 * z0 + qnorm(1 - alpha / 2))} quantiles of
      \code{NMCpop$sample}, where z0 is the proportion of
      \code{NMCpop$sample < NMCpop$mean}.
     \item{\code{hpdCI}} \code{1 - alpha} credible intervals for network
      migratory connectivity diversity, estimated using the highest posterior
      density (HPD) method.
     \item{\code{median}} Medians of network migratory connectivity diversity,
      alternate point estimate also including parametric uncertainty.
     \item{\code{point}} Simple point estimates of network migratory
      connectivity diversity, using the point estimate of \code{psi} (usually
      the mean values), not accounting for sampling error.
   }
  }
  \item{\code{NMCa}}{If parameter \code{originRelAbund} is entered, a list
   containing estimates of abundance-weighted network migratory
   connectivity strength. This list has the same items as NMC, but possibly
   different values.}
  \item{\code{input}}{List containing the inputs to \code{estNMC}.}
}
}
\description{
Resampling of uncertainty for NMC_XY (network migratory connectivity
strength between seasons X and Y), network migratory connectivity
diversity (X node-specific version of NMC_XY), and NMCa_XY
(abundance-weighted network migratory connectivity strength) from estimates
of psi (transition probabilities). Psi estimates can come from an
estMigConnectivity object, an RMark psi matrix, MCMC samples, or other
samples expressed in array form.
}
\examples{
\donttest{
  set.seed(101)
  # Uncertainty in detection (RMark estimates)
  # Number of resampling iterations for generating confidence intervals
  nSamplesCMR <- 100
  nSimulationsCMR <- 10

  # the second intermediate psi scenario, the "low" level
  psiTrue <- samplePsis[["Low"]]
  originRelAbundTrue <- rep(0.25, 4)
  trueNMC <- calcNMC(psiTrue, originRelAbund = originRelAbundTrue)
  trueNMC

  # Storage matrix for samples
  cmrNMCSample <- matrix(NA, nSamplesCMR, nSimulationsCMR)
  summaryCMR <- data.frame(Simulation = 1:nSimulationsCMR, True=trueNMC$NMC,
                           mean=NA, se=NA, lcl=NA, ucl=NA)
  # Get 'RMark' psi estimates and estimate MC from each
  for (r in 1:nSimulationsCMR) {
    cat("Simulation",r,"of",nSimulationsCMR,"\n")
    # Note: getCMRexample() requires a valid internet connection and that GitHub
    # is accessible
    fm <- getCMRexample(r)
    results <- estNMC(psi = fm,
                      originSites = 5:8, targetSites = c(3,2,1,4),
                      nSamples = nSamplesCMR, verbose = 0)
    cmrNMCSample[ , r] <- results$NMC$sample
    summaryCMR$mean[r] <- results$NMC$mean
    summaryCMR$se[r] <- results$NMC$se
    # Calculate confidence intervals using quantiles of sampled MC
    summaryCMR[r, c('lcl', 'ucl')] <- results$NMC$simpleCI
  }

  summaryCMR <- transform(summaryCMR, coverage = (True>=lcl & True<=ucl))
  summaryCMR
  summary(summaryCMR)
  biasCMR <- mean(summaryCMR$mean) - trueNMC$NMC
  biasCMR
  mseCMR <- mean((summaryCMR$mean - trueNMC$NMC)^2)
  mseCMR
  rmseCMR <- sqrt(mseCMR)
  rmseCMR

  # Simulation of BBS data to quantify uncertainty in relative abundance
  nSamplesAbund <- 700 #1700 are stored
  nSimulationsAbund <- 10
  #\dontrun{
  #  nSamplesAbund <- 1700
  #}
  # Storage matrix for samples
  abundNMCaSample <- matrix(NA, nSamplesAbund, nSimulationsAbund)
  summaryAbund <- data.frame(Simulation = 1:nSimulationsAbund,
                             True = trueNMC$NMCa,
                             mean = NA, se = NA, lcl = NA, ucl = NA)
  for (r in 1:nSimulationsAbund) {
    cat("Simulation",r,"of",nSimulationsAbund,"\n")
    row0 <- nrow(abundExamples[[r]]) - nSamplesAbund
    results <- estNMC(originRelAbund = abundExamples[[r]], psi = psiTrue,
                           row0 = row0, nSamples = nSamplesAbund, verbose = 2)
    abundNMCaSample[ , r] <- results$NMCa$sample
    summaryAbund$mean[r] <- results$NMCa$mean
    summaryAbund$se[r] <- results$NMCa$se
    # Calculate confidence intervals using quantiles of sampled MC
    summaryAbund[r, c('lcl', 'ucl')] <- results$NMCa$simpleCI
  }

  summaryAbund <- transform(summaryAbund, coverage = (True >= lcl & True <= ucl))
  summaryAbund
  summary(summaryAbund)
  biasAbund <- mean(summaryAbund$mean) - trueNMC$NMCa
  biasAbund
  mseAbund <- mean((summaryAbund$mean - trueNMC$NMCa)^2)
  mseAbund
  rmseAbund <- sqrt(mseAbund)
  rmseAbund

  # Ovenbird example with GL and GPS data
  data(OVENdata) # Ovenbird

  nSamplesGLGPS <- 100 # Number of bootstrap iterations, set low for example

  # Estimate transition probabilities
  Combined.psi<-estTransition(isGL=OVENdata$isGL, #Light-level geolocator (T/F)
                              isTelemetry = !OVENdata$isGL,
                  geoBias = OVENdata$geo.bias, # Light-level GL location bias
                  geoVCov = OVENdata$geo.vcov, # Location covariance matrix
                  targetSites = OVENdata$targetSites, # Nonbreeding/target sites
                  originSites = OVENdata$originSites, # Breeding/origin sites
                  originPoints = OVENdata$originPoints, # Capture Locations
                  targetPoints = OVENdata$targetPoints, #Device target locations
                  verbose = 3,   # output options
                  nSamples = nSamplesGLGPS, # This is set low for example
                  resampleProjection = sf::st_crs(OVENdata$targetPoints),
                  nSim = 1000)

  # Can estimate NMC from previous psi estimate
  Combo.NMC1 <- estNMC(psi = Combined.psi,
                       nSamples = nSamplesGLGPS)
  Combo.NMC1

  # Doesn't have to be an estPsi object - can simply be array of psi samples
  Combo.MC2 <- estNMC(psi = Combined.psi$psi$sample, # Array of samples
                      originNames = Combined.psi$input$originNames,
                      nSamples = nSamplesGLGPS)
  Combo.MC2

  # Can estimate NMC from previous psi estimate and abundance estimate
  Combo.NMC3 <- estNMC(psi = Combined.psi,
                       originRelAbund = OVENdata$originRelAbund,
                       nSamples = nSamplesGLGPS)
  Combo.NMC3
}
}
\seealso{
\code{\link{calcNMC}}, \code{\link{estTransition}},
  \code{\link{estStrength}}, \code{\link{estMantel}},
  \code{\link{plot.estMigConnectivity}}
}
